#
# This file is part of Python Terra
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#

import evas.decorators

import base

__all__ = ("Screen",)

class Screen(base.EdjeWidget):
    """Base screen to be used by MainWindow.

    This represents a screen that will be swallowed by
    L{window.MainWindow}. It is a L{base.EdjeWidget}, that means that
    it will change on L{theme_changed()}.

    It should provide a L{title} attribute and the obligatory
    L{transition_to()} and L{transition_from()} methods.

    The method B{transition_finished()} notifies when the screen
    transition ends. It should be overridden.

    The method B{has_options()} returns whether Options button should
    be shown or not. If shown, method B{options()} will be called by
    L{window.MainWindow} when the button is clicked.

    L{view_name} class attribute will be used by default L{transition_to()}
    and L{transition_from()} as the "source" of Edje signal emission.

    It can override the methods L{handle_key_up()} and L{handle_key_down()}
    to intercept keyboard events. These methods should return False (which
    is default behaviour) if they want to repass to parent widget.

    """
    view_name = None

    def __init__(self, canvas, group, main_window, title, theme=None):
        self._title = title
        base.EdjeWidget.__init__(self, canvas, group, main_window, theme)

    def title_get(self):
        return self._title

    def title_set(self, title):
        self._title = title
        self._parent_widget.title_replace(title)

    title = property(title_get, title_set)

    def transition_to(self, new_view, end_callback=None):
        if end_callback is not None:
            def cb(*ignored):
                self.signal_callback_del("transition,out,finished", "", cb)
                end_callback(self, new_view)

            self.signal_callback_add("transition,out,finished", "", cb)

        self.signal_emit("transition,out", new_view.view_name)

    def transition_from(self, old_view, end_callback=None):
        if end_callback is not None:
            def cb(*ignored):
                self.signal_callback_del("transition,in,finished", "", cb)
                end_callback(self, old_view)

            self.signal_callback_add("transition,in,finished", "", cb)

        self.signal_emit("transition,in", old_view.view_name)

    def transition_finished(self, old_view, new_view):
        pass

    def has_options(self):
        return False

    def custom_options_button(self):
        return None

    def custom_back_button(self):
        return None

    @evas.decorators.key_up_callback
    def _cb_on_key_up(self, event):
        if not self.handle_key_up(event):
            self._parent_widget.handle_key_up(event)

    def handle_key_up(self, event):
        pass

    @evas.decorators.key_down_callback
    def _cb_on_key_down(self, event):
        if not self.handle_key_down(event):
            self._parent_widget.handle_key_down(event)

    def handle_key_down(self, event):
        pass
