/*
 *
 *  Quick Controller
 *
 *  Copyright (C) 2008-2009  Intel Corporation. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 
 * 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

/****************************************************************************
File Name:          CMPC_QCPLinuxIF.h

Description:
    This file implements to call the operation system and the Device Control
    Service interfaces.
        This QCPVolumeIF class has following functions.

         1.QCP_GetVolume         Get current volume
         2.QCP_SetVolume         Set the volume
         3.QCP_SetVolumeMute     Set the mute

         This CQCPScreenRotation class has following property:

         1.QCP_RotationDegreeToIndexCW   Get the rotation degree of left
         2.QCP_RotationDegreeToIndexCCW  Get the rotation degree of right

        This QCPLinuxIF class has following property:

         1.QCP_InitDCS                   Initialize the Device Control Service
         2.QCP_InitVolumeStatus          Get initalizal volume information
         3.QCP_InitBrightnessStatus      Get initalizal brightness information
         4.QCP_InitRotation              Launch autu rotation(Default mode)
         5.QCP_LCDBrightnessUp           Set Brightness up
         6.QCP_LCDBrightnessDown         Set Brightness down
         7.QCP_VolumeUp                  Set Volume up
         8.QCP_VolumeDown                Set Volume down
         9.QCP_VolumeMute                Set Mute mode
         10.QCP_ShowCursor               Launch the show cursor tool
         11.QCP_LaunchDisplayApp         Launch the Switch Display tool
         12.QCP_LaunchScreenCalApp       Launch the Screen calibrate
         13.QCP_ScreenRotationSwitch     Switch rotation mode
         14.QCP_SetCW                    Right direction rotation
         15.QCP_SetCCW                   Left direction rotation

Environment (opt):
    OS: Ubuntu with latest service pack
    SE: CodeBlocks
Notes (opt):
  =====================================================================
  Revision   Revision History               Author     Date
  =====================================================================

  0.1        Create                       Chen Dequan   2008-7-20

  =====================================================================

****************************************************************************/
#include "CMPC_QCPLinuxIF.h"
#if defined(__UNIX__)

#include "LinuxHideCursor.h"

#ifdef QCP_DEBUG
#include <fstream>
using namespace std;
ofstream dcsLog2;
#endif //QCP_DEBUG

/**************************************************************************************************
* Function Name : QCPInterface                                                     *
* Description  	: Init for QCP                                                                    *
* Date   	: 08/12/01                                                                            *
* Parameter     : [[EN]] None [[EN]]                                                              *
* Return Code  	:                                                                                 *
* Author   	:                                                                                     *
**************************************************************************************************/
QCPInterface::QCPInterface()
    :m_pVolumeIF(NULL),
    m_pScreenRotation(NULL),
    m_pDCS(NULL)
{
    m_pVolumeIF = new QCPVolumeIF();

    m_iCurrentBright = 0;        //Getting current brightness
    m_bStatus    = AUTOROTATION_DISABLED;  //Default auto rotation mode
    m_bMuteState = false;            	 //Default Un-mute mode
    m_iRight = 0;
    m_iLeft  = 0;
    m_iCurrentValueOfVolume = 0; //Current value of volume
    m_iVolumeCopy      = 0; //Backup current value of volume
    m_iVolumeMaxValue  = 25700; //Max value of volume
    m_iCurrentLevel    = 0; //Current level of volume

    m_iVolLevel        = 2;
    m_iVolumeLevelONOF = SOUND_ONOFF_MIN;//Volume level
    m_iVolChangeLev    = ((m_iVolumeLevelONOF / m_iVolLevel) << 8) + (m_iVolumeLevelONOF / m_iVolLevel);
}

/**************************************************************************************************
* Function Name : ~QCPInterface                                                     *
* Description  	: Init for QCP                                                                    *
* Date   	: 08/12/01                                                                            *
* Parameter     : [[EN]] None [[EN]]                                                              *
* Return Code  	:                                                                                 *
* Author   	:                                                                                     *
**************************************************************************************************/
QCPInterface::~QCPInterface()
{
    if ( NULL != m_pVolumeIF )
    {
        delete m_pVolumeIF;
        m_pVolumeIF       = NULL;
    }
    if ( NULL != m_pScreenRotation )
    {
        delete m_pScreenRotation;
        m_pScreenRotation = NULL;
    }
    if ( NULL != m_pDCS )
    {
        delete m_pDCS;
        m_pDCS = NULL;
    }
}

/**************************************************************************************************
* Function Name : QCP_InitDCS                                                     *
* Description  	: Init for QCP                                                                    *
* Date   	: 08/12/01                                                                            *
* Parameter     : [[EN]] None [[EN]]                                                              *
* Return Code  	:                                                                                 *
* Author   	:                                                                                     *
**************************************************************************************************/
bool QCPInterface::QCP_InitDCS()
{
    m_pDCS = new CDCSBrightRotation();
    if ( NULL == m_pDCS )
    {
        return false;
    }

    if ( m_pDCS->InitDCS() )
    {//If the device control service initialized successfully, return true
        return true;
    }

    return false;
}

/**************************************************************************************************
* Function Name : QCP_InitBrightnessStatus                                                     *
* Description  	: Init for QCP                                                                    *
* Date   	: 08/12/01                                                                            *
* Parameter     : [[EN]] None [[EN]]                                                              *
* Return Code  	:                                                                                 *
* Author   	:                                                                                     *
**************************************************************************************************/
bool QCPInterface::QCP_InitBrightnessStatus(int *piBrightLevel)
{
    if ( (NULL == piBrightLevel) || (NULL == m_pDCS) )
    {
        return false;
    }

    if ( true == m_pDCS->InitBrightness(&m_iCurrentBright) )
    {//If the brightness initialized successfully, return current brightness level and true
        *piBrightLevel = m_iCurrentBright;
        return true;
    }

     //Return init brightness level when adjustment the brightness failed
    *piBrightLevel = m_iCurrentBright;
    return false;
}

/**************************************************************************************************
* Function Name : QCP_LCDBrightnessUp                                                     *
* Description  	: Init for QCP                                                                    *
* Date   	: 08/12/01                                                                            *
* Parameter     : [[EN]] None [[EN]]                                                              *
* Return Code  	:                                                                                 *
* Author   	:                                                                                     *
**************************************************************************************************/
bool QCPInterface::QCP_LCDBrightnessUp(int iBrightLevel, int *piBrightLevel)
{
    if ( (NULL == piBrightLevel) || (NULL == m_pDCS) )
    {
        return false;
    }

    if ( true == m_pDCS->LCDBrightnessUp(iBrightLevel, &m_iCurrentBright) )
    {//If brightness up successfully, return brightness level and true
        *piBrightLevel = m_iCurrentBright; //Return current brightntness level
        return true;
    }

     //Return last time brightness level when adjustment the brightness failed
    *piBrightLevel = m_iCurrentBright;
    return false;
}

/**************************************************************************************************
* Function Name : QCP_LCDBrightnessDown                                                     *
* Description  	: Init for QCP                                                                    *
* Date   	: 08/12/01                                                                            *
* Parameter     : [[EN]] None [[EN]]                                                              *
* Return Code  	:                                                                                 *
* Author   	:                                                                                     *
**************************************************************************************************/
bool QCPInterface::QCP_LCDBrightnessDown(int iBrightLevel, int *piBrightLevel)
{
    if ( (NULL == piBrightLevel) || (NULL == m_pDCS) )
    {
        return false;
    }

    if ( true == m_pDCS->LCDBrightnessDown(iBrightLevel, &m_iCurrentBright) )
    {//If brightness down successfully, return brightness level and true
        *piBrightLevel = m_iCurrentBright;//Return current brightntness level
        return true;
    }

    //Return last time brightness level when adjustment the brightness failed
    *piBrightLevel = m_iCurrentBright;
	return false;
}

/**************************************************************************************************
* Function Name : QCP_SetLCDBrightnessLevel                                                     *
* Description  	: Init for QCP                                                                    *
* Date   	: 08/12/01                                                                            *
* Parameter     : [[EN]] None [[EN]]                                                              *
* Return Code  	:                                                                                 *
* Author   	:                                                                                     *
**************************************************************************************************/
bool QCPInterface::QCP_SetLCDBrightnessLevel(int iBrightLevel)
{
    if ( NULL == m_pDCS )
    {
        return false;
    }

    //Setting brightness level when user click the progress bar
    if (true == m_pDCS->SetLCDBrightness(iBrightLevel))
    {
        return true;
    }

    return false;
}

/**************************************************************************************************
* Function Name : QCP_InitVolumeStatus                                                     *
* Description  	: Init for QCP                                                                    *
* Date   	: 08/12/01                                                                            *
* Parameter     : [[EN]] None [[EN]]                                                              *
* Return Code  	:                                                                                 *
* Author   	:                                                                                     *
**************************************************************************************************/
bool QCPInterface::QCP_InitVolumeStatus(int *piVolLevel, bool* pbMuteState)
{
    #ifdef QCP_DEBUG
    dcsLog2.open("/root/QCP_Log.txt", ios_base::app);
    dcsLog2 << "QCP_InitVolumeStatus enter " << endl;
    #endif
    if ( (NULL == piVolLevel)
        || (NULL == pbMuteState)
        || (NULL == m_pVolumeIF) )
    {
        return false;
    }

    if ( false == m_pVolumeIF->QCP_GetVolume(&m_iCurrentValueOfVolume) )
    {
        //If the current volume haven't been got, return false
        return false;
    }
    if ( 0 == m_iCurrentValueOfVolume ) //If OS initializtion is mute
    {
        m_iVolumeCopy = VOLUME_MIN; //Set the min of volume
        m_bMuteState = true;
        *pbMuteState = true;
    }
    else
    {
        m_iVolumeCopy = m_iCurrentValueOfVolume;
        m_bMuteState = false;
        *pbMuteState = false;
    }
    m_iCurrentLevel = (m_iCurrentValueOfVolume + 257) / m_iVolChangeLev;//Get level of volume
    #ifdef QCP_DEBUG
        dcsLog2 << "m_iCurrentValueOfVolume: " << m_iCurrentValueOfVolume << endl;
        dcsLog2 << "m_iCurrentLevel: " << m_iCurrentLevel << endl;
    #endif
    if ((m_iCurrentLevel == VOLUMELEVEL) && (m_iCurrentValueOfVolume < m_iVolumeMaxValue))
    {
        m_iCurrentLevel = VOLUMELEVEL - 1;
    }
    if ((m_iCurrentLevel == 0) && (m_iCurrentValueOfVolume > VOLUME_MIN))
    {
        m_iCurrentLevel = 1;
    }
    *piVolLevel = m_iCurrentLevel;//Return current volume level for update UI
    #ifdef QCP_DEBUG
        dcsLog2 << "End Sync" << m_iCurrentValueOfVolume << endl;
        dcsLog2.close();
    #endif

    return true;
}

/**************************************************************************************************
* Function Name : QCP_VolumeUp                                                     *
* Description  	: Init for QCP                                                                    *
* Date   	: 08/12/01                                                                            *
* Parameter     : [[EN]] None [[EN]]                                                              *
* Return Code  	:                                                                                 *
* Author   	:                                                                                     *
**************************************************************************************************/
bool QCPInterface::QCP_VolumeUp(int *piVolLevel)
{
    if ( ( m_bMuteState )  //If current volume is mute mode, the volume can't adjusted
        || (NULL == piVolLevel)
        || (NULL == m_pVolumeIF ) )
    {
        return false;
    }

    if (m_iCurrentLevel == (VOLUMELEVEL - 1))
    {
        if ( false == m_pVolumeIF->QCP_SetVolume(m_iVolumeMaxValue) ) //Setting max volume
        {
            return false;
        }
        m_iVolumeCopy = m_iVolumeMaxValue;
        m_iCurrentLevel = VOLUMELEVEL;
        *piVolLevel = m_iCurrentLevel; //Return the volume level to update UI
        return true;
    }

    if ( false == m_pVolumeIF->QCP_GetVolume(&m_iCurrentValueOfVolume) )//Getting current volume
    {
        return false;
    }
    //Getting left and right track
    m_iLeft  = m_iCurrentValueOfVolume & 0xFF;
    m_iRight = (m_iCurrentValueOfVolume >> 8) & 0xFF;

    //Up left and right track
    m_iLeft  += m_iVolumeLevelONOF / m_iVolLevel;
    m_iRight += m_iVolumeLevelONOF / m_iVolLevel;

    m_iCurrentValueOfVolume = (m_iRight << 8) + m_iLeft;
    m_iVolumeCopy = m_iCurrentValueOfVolume;   //Backup volume for the mute mode switch

    if ( false == m_pVolumeIF->QCP_SetVolume(m_iCurrentValueOfVolume) ) //Setting volume
    {
        return false;
    }

    m_iCurrentLevel = m_iCurrentValueOfVolume / m_iVolChangeLev;
    *piVolLevel = m_iCurrentLevel; //Return the volume level to update UI

    return true;
}

/**************************************************************************************************
* Function Name : QCP_VolumeDown                                                     *
* Description  	: Init for QCP                                                                    *
* Date   	: 08/12/01                                                                            *
* Parameter     : [[EN]] None [[EN]]                                                              *
* Return Code  	:                                                                                 *
* Author   	:                                                                                     *
**************************************************************************************************/
bool QCPInterface::QCP_VolumeDown(int *piVolLevel)
{
    if ( (m_bMuteState) //If current volume is mute mode, the volume can't adjusted
        || (NULL == piVolLevel)
        || (NULL == m_pVolumeIF) )
    {
        return false;
    }

     //Getting current volume level
    if ( false == m_pVolumeIF->QCP_GetVolume(&m_iCurrentValueOfVolume) )
    {
        return false;
    }
    if ( m_iCurrentValueOfVolume >= m_iVolChangeLev )
    {
        //Getting left and right track
        m_iLeft  = m_iCurrentValueOfVolume & 0xFF;
        m_iRight = (m_iCurrentValueOfVolume >> 8) & 0xFF;

        //Down left and right track
        m_iRight -= m_iVolumeLevelONOF / m_iVolLevel;
        m_iLeft  -= m_iVolumeLevelONOF / m_iVolLevel;

        m_iCurrentValueOfVolume = (m_iRight << 8) + m_iLeft;
        m_iVolumeCopy = m_iCurrentValueOfVolume;                 //Backup volume for the mute mode switch
        *piVolLevel = m_iCurrentValueOfVolume / m_iVolChangeLev; //Return volume level to update UI
        if (*piVolLevel == 0)
        {
            m_iCurrentValueOfVolume = VOLUME_MIN; //Setting min volume
            m_iVolumeCopy  = VOLUME_MIN;
        }
    }
    else if ( m_iCurrentValueOfVolume < m_iVolChangeLev )
    {
        m_iCurrentValueOfVolume = VOLUME_MIN; //Setting min volume
        m_iVolumeCopy  = VOLUME_MIN;
        *piVolLevel    = 0;          //Return 0 level to update UI
    }
    if ( false == m_pVolumeIF->QCP_SetVolume(m_iCurrentValueOfVolume) )//Setting volume
    {
        return false;
    }

    return true;
}

/**************************************************************************************************
* Function Name : QCP_SetVolumeLevel                                                     *
* Description  	: Init for QCP                                                                    *
* Date   	: 08/12/01                                                                            *
* Parameter     : [[EN]] None [[EN]]                                                              *
* Return Code  	:                                                                                 *
* Author   	:                                                                                     *
**************************************************************************************************/
bool QCPInterface::QCP_SetVolumeLevel(int iVolLevel)
{
    #ifdef QCP_DEBUG
    dcsLog2.open("/root/QCP_Log.txt", ios_base::app);
    dcsLog2 << "Set Volume Level: iVolLevel: " << iVolLevel << endl;
    #endif

    if ( (m_bMuteState) //If current volume is mute mode, the volume can't adjusted
         || (NULL == m_pVolumeIF) )
    {
        return false;
    }

    if ( VOLUMELEVEL == iVolLevel )  // 7 level
    {
        m_iVolumeCopy = m_iVolumeMaxValue; //Setting max volume
        m_iCurrentValueOfVolume = m_iVolumeMaxValue;
        m_iCurrentLevel = VOLUMELEVEL;
    }
    else if ( 0 == iVolLevel )  // 0 level
    {
        m_iVolumeCopy = 257; //Setting max volume
        m_iCurrentValueOfVolume = 257;
        m_iCurrentLevel = 0;
    }
    else
    {
        m_iCurrentValueOfVolume = m_iVolChangeLev * iVolLevel;//Get current value of volume
        m_iVolumeCopy = m_iCurrentValueOfVolume;//Control level of volume under mute
        m_iCurrentLevel = iVolLevel;
        #ifdef QCP_DEBUG
        dcsLog2 << "m_iCurrentValueOfVolume: " << m_iCurrentValueOfVolume << endl;
        dcsLog2 << "m_iVolumeCopy: " << m_iVolumeCopy << endl;
        dcsLog2 << "m_iCurrentLevel: " << m_iCurrentLevel << endl;
        #endif
    }
    if ( false == m_pVolumeIF->QCP_SetVolume(m_iCurrentValueOfVolume) )
    {
        return false;
    }
    #ifdef QCP_DEBUG
    dcsLog2 << "Set Volume Level end" << endl;
    dcsLog2.close();
    #endif
    return true;
}

/**************************************************************************************************
* Function Name : QCP_VolumeMute                                                     *
* Description  	: Init for QCP                                                                    *
* Date   	: 08/12/01                                                                            *
* Parameter     : [[EN]] None [[EN]]                                                              *
* Return Code  	:                                                                                 *
* Author   	:                                                                                     *
**************************************************************************************************/
bool QCPInterface::QCP_VolumeMute(int *piCurrentVolLevel, bool* pbStatus)
{
    if ( (NULL == piCurrentVolLevel)
        || (NULL == pbStatus)
        || ( NULL == m_pVolumeIF) )
    {
        return false;
    }

    //Getting current volume mode
    if ( false == m_pVolumeIF->QCP_GetVolume(&m_iCurrentValueOfVolume) )
    {
        return false;
    }

    // Setting mute mode
    if ( false == m_pVolumeIF->QCP_SetVolumeMute(&m_iVolumeCopy, &m_bMuteState, m_iCurrentValueOfVolume) )
    {
        return false;
    }

    //Return volume level and current status under mute mode
    *piCurrentVolLevel = 0;//m_iVolumeCopy / m_iVolChangeLev;
    *pbStatus = m_bMuteState;

    return true;
}

/**************************************************************************************************
* Function Name : QCP_GetCursorStatus                                                     *
* Description  	: Init for QCP                                                                    *
* Date   	: 08/12/01                                                                            *
* Parameter     : [[EN]] None [[EN]]                                                              *
* Return Code  	:                                                                                 *
* Author   	:                                                                                     *
**************************************************************************************************/
bool QCPInterface::QCP_GetCursorStatus()
{
    CHideCursor HideCursor;
    return HideCursor.GetCursorStatus();
}

#include <wx/msgdlg.h>

/**************************************************************************************************
* Function Name : QCP_ShowCursor                                                     *
* Description  	: Init for QCP                                                                    *
* Date   	: 08/12/01                                                                            *
* Parameter     : [[EN]] None [[EN]]                                                              *
* Return Code  	:                                                                                 *
* Author   	:                                                                                     *
**************************************************************************************************/
void QCPInterface::QCP_ShowCursor(wxString AppShowCursor)//Show or hide cursor
{
    string ShowCursor = m_TypeTransfer.fWxStringToString(AppShowCursor);
    CHideCursor HideCursor;
    if (strcmp(ShowCursor.c_str(), "TRUE") == 0)
	{
		HideCursor.SetCursorStatus(TRUE);
		HideCursor.HideCursor();
		//system("/usr/bin/dcs/hidecursor.sh");//Hide the cursor
	}
	else
	{
	 	HideCursor.SetCursorStatus(FALSE);
		HideCursor.ShowCursor();
		//system("/usr/bin/dcs/showcursor.sh");// Show cursor
	}
}

/**************************************************************************************************
* Function Name : QCP_LaunchDisplayApp                                                     *
* Description  	: Init for QCP                                                                    *
* Date   	: 08/12/01                                                                            *
* Parameter     : [[EN]] None [[EN]]                                                              *
* Return Code  	:                                                                                 *
* Author   	:                                                                                     *
**************************************************************************************************/
void QCPInterface::QCP_LaunchDisplayApp(wxString AppDisplay) //Screen Switch
{
    string sAppDisplay = m_TypeTransfer.fWxStringToString(AppDisplay);
    system(sAppDisplay.c_str());
}

/**************************************************************************************************
* Function Name : QCP_LaunchScreenCalApp                                                     *
* Description  	: Init for QCP                                                                    *
* Date   	: 08/12/01                                                                            *
* Parameter     : [[EN]] None [[EN]]                                                              *
* Return Code  	:                                                                                 *
* Author   	:                                                                                     *
**************************************************************************************************/
bool QCPInterface::QCP_LaunchScreenCalApp(wxString AppPath)
{
    string sAppPath = m_TypeTransfer.fWxStringToString(AppPath);
    int iRet = system(sAppPath.c_str());
    if (iRet == 0)
    {
        return true; //The touch screen tool launched successfully
    }
    else
    {
        return false;//fail
    }
}

/**************************************************************************************************
* Function Name : QCP_InitRotation                                                     *
* Description  	: Init for QCP                                                                    *
* Date   	: 08/12/01                                                                            *
* Parameter     : [[EN]] None [[EN]]                                                              *
* Return Code  	:                                                                                 *
* Author   	:                                                                                     *
**************************************************************************************************/
bool QCPInterface::QCP_InitRotation(BOOL* pbStatus)
{
    if ( (NULL == pbStatus) || (NULL == m_pDCS) )
    {
        return false;
    }

    m_pScreenRotation = new CQCPScreenRotation();
    if (NULL == m_pScreenRotation)
    {
        *pbStatus = m_bStatus;//maunal rotation
        return false;
    }

    if ( false == m_pDCS->GetAutoRotationStatus(&m_bStatus) )
    {
        *pbStatus = m_bStatus;//maunal rotation
        return false;
    }

    if (m_bStatus == AUTOROTATION_ENABLED)
    {
        //Current mode is auto rotation
        *pbStatus = AUTOROTATION_ENABLED;  //Update UI
    }
    else
    {
        //Current mode is maunal rotation
        *pbStatus = AUTOROTATION_DISABLED;//Update UI
    }

    return true;
}

/**************************************************************************************************
* Function Name : QCP_ScreenRotationSwitch                                                     *
* Description  	: Init for QCP                                                                    *
* Date   	: 08/12/01                                                                            *
* Parameter     : [[EN]] None [[EN]]                                                              *
* Return Code  	:                                                                                 *
* Author   	:                                                                                     *
**************************************************************************************************/
bool QCPInterface::QCP_ScreenRotationSwitch(BOOL* pbStatus)
{
    if ( (NULL == pbStatus) || (NULL == m_pDCS) )
    {
        return false;
    }

    if ( AUTOROTATION_DISABLED == m_bStatus )           //Switch rotation mode
    {
        if ( true == m_pDCS->EnableAutoRotation(&m_bStatus))
        {
            if ( AUTOROTATION_ENABLED == m_bStatus )
            {                           //Current mode switch to  auto rotation
                *pbStatus = AUTOROTATION_ENABLED;  //Update UI
                return true;
            }
            else
            {
                *pbStatus = AUTOROTATION_DISABLED;  //Update UI
                return false;
            }
        }
        else
        {
            *pbStatus = AUTOROTATION_DISABLED;  //Update UI
            return false;
        }
    }
    else  // Switch to maunal rotation mode
    {
        if ( true == m_pDCS->DisableAutoRotation(&m_bStatus) )
        {
            if ( AUTOROTATION_DISABLED == m_bStatus )
            {                         //Current mode switch to maunal rotation
                *pbStatus = AUTOROTATION_DISABLED;//Update UI
                return true;
            }
            else
            {
                *pbStatus = AUTOROTATION_ENABLED;  //Update UI
                return false;
            }
        }
        else
        {
            *pbStatus = AUTOROTATION_ENABLED;  //Update UI
            return false;
        }
    }

    return false;
}

/**************************************************************************************************
* Function Name : QCP_SetCW                                                     *
* Description  	: Init for QCP                                                                    *
* Date   	: 08/12/01                                                                            *
* Parameter     : [[EN]] None [[EN]]                                                              *
* Return Code  	:                                                                                 *
* Author   	:                                                                                     *
**************************************************************************************************/
void QCPInterface::QCP_SetCW(CMPC_QCPDialog* pDlg)
{
    if ( (NULL == pDlg) || (NULL == m_pScreenRotation) )
    {
        return;
    }

    m_CurrentRotation = RR_Rotate_0;
    if ( false == m_pScreenRotation->QCP_GetRotationInfo(m_CurrentRotation) )//Get current rotation degree
    {
        return;
    }
    m_pScreenRotation->QCP_RotationDegreeToIndexCW(&m_Rotation, m_CurrentRotation);//Get rotation degree
    m_pScreenRotation->QCP_Rotation(m_Rotation); //Begin rotation

    return;
}

/**************************************************************************************************
* Function Name : QCP_SetCCW                                                     *
* Description  	: Init for QCP                                                                    *
* Date   	: 08/12/01                                                                            *
* Parameter     : [[EN]] None [[EN]]                                                              *
* Return Code  	:                                                                                 *
* Author   	:                                                                                     *
**************************************************************************************************/
void QCPInterface::QCP_SetCCW(CMPC_QCPDialog* pDlg)
{
    if ( (NULL == pDlg) || (NULL == m_pScreenRotation) )
    {
        return;
    }

    m_CurrentRotation = RR_Rotate_0;
    if ( false == m_pScreenRotation->QCP_GetRotationInfo(m_CurrentRotation) ) //Get current degree
    {
        return;
    }
    m_pScreenRotation->QCP_RotationDegreeToIndexCCW(&m_Rotation, m_CurrentRotation); //Get rotation degree
    m_pScreenRotation->QCP_Rotation(m_Rotation); //Begin rotation

    return;
}

/**************************************************************************************************
* Function Name : QCP_Rotation180Degree                                                     *
* Description  	: Init for QCP                                                                    *
* Date   	: 08/12/01                                                                            *
* Parameter     : [[EN]] None [[EN]]                                                              *
* Return Code  	:                                                                                 *
* Author   	:                                                                                     *
**************************************************************************************************/
void QCPInterface::QCP_Rotation180Degree(CMPC_QCPDialog* pDlg)
{
    if ( (NULL == pDlg) || (NULL == m_pScreenRotation) )
    {
        return;
    }

    m_CurrentRotation = RR_Rotate_0;
    if ( false == m_pScreenRotation->QCP_GetRotationInfo(m_CurrentRotation) ) //Get current degree
    {
        return;
    }
    m_pScreenRotation->QCP_Rotation180DegreeToIndex(&m_Rotation, m_CurrentRotation); //Get rotation degree
    m_pScreenRotation->QCP_Rotation(m_Rotation); //Begin rotation

    return;
}

/**************************************************************************************************
* Function Name : QCPVolumeIF                                                     *
* Description  	: Init for QCP                                                                    *
* Date   	: 08/12/01                                                                            *
* Parameter     : [[EN]] None [[EN]]                                                              *
* Return Code  	:                                                                                 *
* Author   	:                                                                                     *
**************************************************************************************************/
QCPVolumeIF::QCPVolumeIF()
{
    m_iLeft = 0; //Left volume channel
    m_iRight = 0; //Right volume channel
    m_iCurrentVolume = 0;
    m_iMute = 0x0000;  //Setting mute
}

/**************************************************************************************************
* Function Name : ~QCPVolumeIF                                                     *
* Description  	: Init for QCP                                                                    *
* Date   	: 08/12/01                                                                            *
* Parameter     : [[EN]] None [[EN]]                                                              *
* Return Code  	:                                                                                 *
* Author   	:                                                                                     *
**************************************************************************************************/
QCPVolumeIF::~QCPVolumeIF()
{
}

/**************************************************************************************************
* Function Name : QCP_GetVolume                                                     *
* Description  	: Init for QCP                                                                    *
* Date   	: 08/12/01                                                                            *
* Parameter     : [[EN]] None [[EN]]                                                              *
* Return Code  	:                                                                                 *
* Author   	:                                                                                     *
**************************************************************************************************/
bool QCPVolumeIF::QCP_GetVolume(int* iCurrentVolLevel)
{
    if ( NULL == iCurrentVolLevel )
    {
        return false;
    }
    int iMixer = open(VOLUME_DEVICE, O_RDONLY);//Open the mixer device
    if ( iMixer < 0 )
    {
        return false;	//Can't open the Mixer device
    }
    if (-1 == ioctl(iMixer, SOUND_MIXER_READ_VOLUME, &m_iCurrentVolume))//Get current volume value
    {
        close(iMixer);	//Close the mixer device
        return false;
    }
    close(iMixer);		//Close the mixer device

    m_iLeft = m_iCurrentVolume & 0xFF;
    m_iRight = (m_iCurrentVolume >> 8) & 0xFF;
    if(m_iLeft != m_iRight)
    {
        m_iLeft = m_iRight = (m_iLeft + m_iRight) / 2;
    }
    *iCurrentVolLevel = (m_iRight << 8) + m_iLeft; //Return current volume level

    return true;
}

/**************************************************************************************************
* Function Name : QCP_SetVolume                                                     *
* Description  	: Init for QCP                                                                    *
* Date   	: 08/12/01                                                                            *
* Parameter     : [[EN]] None [[EN]]                                                              *
* Return Code  	:                                                                                 *
* Author   	:                                                                                     *
**************************************************************************************************/
bool QCPVolumeIF::QCP_SetVolume(int iVolumeLevel)
{
    int iMixer = open(VOLUME_DEVICE, O_WRONLY); //Opening the mixer device
    if ( iMixer < 0 )
    {
        return false;                          //can't open mixer device
    }
    if ( -1 == ioctl(iMixer, SOUND_MIXER_WRITE_VOLUME, &iVolumeLevel) ) //Set volume value
    {
        close(iMixer);		//Close the mixer device
        return false;
    }
    close(iMixer);

    return true;
}

/**************************************************************************************************
* Function Name : QCP_SetVolumeMute                                                     *
* Description  	: Init for QCP                                                                    *
* Date   	: 08/12/01                                                                            *
* Parameter     : [[EN]] None [[EN]]                                                              *
* Return Code  	:                                                                                 *
* Author   	:                                                                                     *
**************************************************************************************************/
bool QCPVolumeIF::QCP_SetVolumeMute(int* piVolumeCopy, bool* pbMode, int iCurrentVolLevel)
{
    if ( (NULL == piVolumeCopy) || (NULL == pbMode) )
    {
        return false;
    }

    /*Open the mixer device*/
    int iMixer = open(VOLUME_DEVICE, O_WRONLY);
    if ( iMixer < 0 )
    {
        return false;
    }

    if ( m_iMute != iCurrentVolLevel ) //Setting mute mode
    {
        //*piVolumeCopy = iCurrentVolLevel;
        //*piVolumeCopy = 0;
        *piVolumeCopy = VOLUME_MIN;

        if ( -1 == ioctl(iMixer, SOUND_MIXER_WRITE_VOLUME, piVolumeCopy) )
        {  //The mute mode set fail
            close(iMixer);//Close the mixer device
            return false;
        }
        wxMilliSleep(100);
        *piVolumeCopy = 0;
        m_iMute = 0x0000;
        if ( -1 == ioctl(iMixer, SOUND_MIXER_WRITE_VOLUME, &m_iMute) )
        {  //The mute mode set fail
            close(iMixer);//Close the mixer device
            return false;
        }
        *pbMode = true;   //Return current status of volume: Mute
    }
    else // Setting Un-mute mode
    {
         //Setting current volume value
        *piVolumeCopy = VOLUME_MIN;
        if ( -1 == ioctl(iMixer, SOUND_MIXER_WRITE_VOLUME, piVolumeCopy) )
        {
            close(iMixer);//Close the mixer device
            return false;
        }
        *pbMode = false;  //Return current status of volume: Unmute
    }
    close(iMixer);

    return true;
}

/**************************************************************************************************
* Function Name : CQCPScreenRotation                                                     *
* Description  	: Init for QCP                                                                    *
* Date   	: 08/12/01                                                                            *
* Parameter     : [[EN]] None [[EN]]                                                              *
* Return Code  	:                                                                                 *
* Author   	:                                                                                     *
**************************************************************************************************/
CQCPScreenRotation::CQCPScreenRotation()
{
}

/**************************************************************************************************
* Function Name : ~CQCPScreenRotation                                                     *
* Description  	: Init for QCP                                                                    *
* Date   	: 08/12/01                                                                            *
* Parameter     : [[EN]] None [[EN]]                                                              *
* Return Code  	:                                                                                 *
* Author   	:                                                                                     *
**************************************************************************************************/
CQCPScreenRotation::~CQCPScreenRotation()
{
}

/**************************************************************************************************
* Function Name : QCP_GetRotationInfo                                                     *
* Description  	: Init for QCP                                                                    *
* Date   	: 08/12/01                                                                            *
* Parameter     : [[EN]] None [[EN]]                                                              *
* Return Code  	:                                                                                 *
* Author   	:                                                                                     *
**************************************************************************************************/
bool CQCPScreenRotation::QCP_GetRotationInfo(Rotation& rCurrentRotation)
{
    m_pDisplay = XOpenDisplay(NULL);
    if ( NULL == m_pDisplay )
    {
        return false;
    }
    XLockDisplay(m_pDisplay);

    m_iScreen = DefaultScreen(m_pDisplay);//Getting Current Screen
    if ( m_iScreen >= ScreenCount(m_pDisplay) )
    {
        XUnlockDisplay(m_pDisplay);
        XCloseDisplay(m_pDisplay);     //Close the display
        return false;                     //Invalid screen number
    }
    m_WinRoot = RootWindow(m_pDisplay, m_iScreen);
    //Getting configure information
    m_pConfig = XRRGetScreenInfo(m_pDisplay, m_WinRoot);
    // Getting current rotation and sizeID
    m_iSizeID = XRRConfigCurrentConfiguration(m_pConfig, &rCurrentRotation);

    return true;
}

/**************************************************************************************************
* Function Name : QCP_Rotation                                                     *
* Description  	: Init for QCP                                                                    *
* Date   	: 08/12/01                                                                            *
* Parameter     : [[EN]] None [[EN]]                                                              *
* Return Code  	:                                                                                 *
* Author   	:                                                                                     *
**************************************************************************************************/
void CQCPScreenRotation::QCP_Rotation(Rotation rRotation)
{
    //Start rotation
    switch (rRotation)
    {
        case RR_Rotate_0:
            //wxMessageBox(wxT("xrandr -o normal"));
            system("xrandr -o normal");
            break;
        case RR_Rotate_90:
            //wxMessageBox(wxT("xrandr -o left"));
            system("xrandr -o left");
            break;
        case RR_Rotate_180:
            //wxMessageBox(wxT("xrandr -o inverted"));
            system("xrandr -o inverted");
            break;
        case RR_Rotate_270:
           // wxMessageBox(wxT("xrandr -o right"));
            system("xrandr -o right");
            break;
        default:
            break;
    }
//    XRRSetScreenConfig(m_pDisplay, m_pConfig, m_WinRoot, m_iSizeID, rRotation, CurrentTime);

    XRRFreeScreenConfigInfo(m_pConfig);
    XUnlockDisplay(m_pDisplay);    //Relaese the lock
    XCloseDisplay(m_pDisplay);     //Close the display
}

/**************************************************************************************************
* Function Name : QCP_RotationDegreeToIndexCW                                                     *
* Description  	: Init for QCP                                                                    *
* Date   	: 08/12/01                                                                            *
* Parameter     : [[EN]] None [[EN]]                                                              *
* Return Code  	:                                                                                 *
* Author   	:                                                                                     *
**************************************************************************************************/
void CQCPScreenRotation::QCP_RotationDegreeToIndexCW(Rotation* piDegreeToIndex, Rotation iDegree)
{
    if ( NULL == piDegreeToIndex )
    {
        return;
    }

    //Get the rotation degree of CW
    switch (iDegree)
    {
        case RR_Rotate_0:
            *piDegreeToIndex = RR_Rotate_90;
            break;
        case RR_Rotate_90:
            *piDegreeToIndex = RR_Rotate_180;
            break;
        case RR_Rotate_180:
            *piDegreeToIndex = RR_Rotate_270;
            break;
        case RR_Rotate_270:
            *piDegreeToIndex = RR_Rotate_0;
            break;
        default:
            break;
    }
}

/**************************************************************************************************
* Function Name : QCP_RotationDegreeToIndexCCW                                                     *
* Description  	: Init for QCP                                                                    *
* Date   	: 08/12/01                                                                            *
* Parameter     : [[EN]] None [[EN]]                                                              *
* Return Code  	:                                                                                 *
* Author   	:                                                                                     *
**************************************************************************************************/
void CQCPScreenRotation::QCP_RotationDegreeToIndexCCW(Rotation* piDegreeToIndex, Rotation iDegree)
{
    if ( NULL == piDegreeToIndex )
    {
        return;
    }

    //Get the rotation degree of CCW
    switch (iDegree)
    {
        case RR_Rotate_0:
            *piDegreeToIndex = RR_Rotate_270;
            break;
        case RR_Rotate_270:
            *piDegreeToIndex = RR_Rotate_180;
            break;
        case RR_Rotate_180:
            *piDegreeToIndex = RR_Rotate_90;
            break;
        case RR_Rotate_90:
            *piDegreeToIndex = RR_Rotate_0;
            break;
        default:
            break;
    }
}

/**************************************************************************************************
* Function Name : QCP_Rotation180DegreeToIndex                                                     *
* Description  	: Init for QCP                                                                    *
* Date   	: 08/12/01                                                                            *
* Parameter     : [[EN]] None [[EN]]                                                              *
* Return Code  	:                                                                                 *
* Author   	:                                                                                     *
**************************************************************************************************/
void CQCPScreenRotation::QCP_Rotation180DegreeToIndex(Rotation* piDegreeToIndex, Rotation iDegree)
{
    if ( NULL == piDegreeToIndex )
    {
        return;
    }

    //Get the rotation degree of CW
    switch (iDegree)
    {
        case RR_Rotate_0:
            *piDegreeToIndex = RR_Rotate_180;
            break;
        case RR_Rotate_90:
            *piDegreeToIndex = RR_Rotate_270;
            break;
        case RR_Rotate_180:
            *piDegreeToIndex = RR_Rotate_0;
            break;
        case RR_Rotate_270:
            *piDegreeToIndex = RR_Rotate_90;
            break;
        default:
            break;
    }
}
#endif

