/*
 * Copyright (c) 2015 Intel Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef _SENSOR_H_
#define _SENSOR_H_

#include "sensors.h"
#include "sensor_device.h"
#include <pthread.h>
#include <vector>
#include <map>

#define US_TO_NS (1000)

typedef enum {
        AXIS_X = 0,
        AXIS_Y = 1,
        AXIS_Z = 2,
        AXIS_MAX,
} sensor_axis_t;

struct sensor_additional_information_t {
        bool expose_to_android;
        bool expose_to_csdk;
        std::vector<float> scale_android;
        std::vector<float> scale_csdk;
        sensor_axis_t axisMap[AXIS_MAX];
        sensor_additional_information_t()
        {
                expose_to_android = false;
                expose_to_csdk = false;
                for (int i = AXIS_X; i < AXIS_MAX; i++)
                        axisMap[i] = static_cast<sensor_axis_t>(i);
        }
};

class Session;

class Sensor {
protected:
        struct sensor_device_t device;
        struct sensor_additional_information_t information;
        struct sensors_event_t * mEvents;
        char * mStreaming;
        size_t mEventsCount;
        size_t mStreamingSize;
        // The same as Android Sensor Service, currently flags is ignored.
        struct BatchParams {
                int64_t batchDelay, batchTimeout;
                BatchParams() : batchDelay(0), batchTimeout(0) {}
                BatchParams(int64_t delay, int64_t timeout): batchDelay(delay), batchTimeout(timeout) { }
                bool operator != (const BatchParams& other) {
                        return other.batchDelay != batchDelay || other.batchTimeout != batchTimeout;
                }
        };
        BatchParams bestBatchParams;
        std::map<Session *, BatchParams> batchParams;
        mutable pthread_mutex_t batchParamsLock; // Lock batchParams for data thread will access it
        int lockbatchParams();
        int unlockbatchParams();
        bool setBatchParamsForIdent(Session * session, int64_t samplingPeriodNs, int64_t maxBatchReportLatencyNs);
        void selectBatchParams();
        size_t removeBatchParamsForIdent(Session * session);
        void reallocStreamingBuffer(size_t size);
        void reallocEventsBuffer(size_t count);
public:
        Sensor(const struct sensor_device_t &device, const sensor_additional_information_t &information);
        Sensor(const Sensor * sensor);
        bool exposeToAndroid() { return information.expose_to_android; }
        bool exposeToCSDK() { return information.expose_to_csdk; }
        const char * getName() { return device.name; }
        const char * getVendor() { return device.vendor; }
        const char * getStringType() { return device.stringType; }
        const char * getRequiredPermission() { return device.requiredPermission; }
        void setName(const char * name) { device.name = const_cast<char *>(name); }
        void setVendor(const char * vendor) { device.vendor = const_cast<char *>(vendor); }
        void setStringType(const char * stringType) { device.stringType = const_cast<char *>(stringType); }
        void setRequiredPermission(const char * requiredPermission) { device.requiredPermission = const_cast<char *>(requiredPermission); }
        int getType() const { return device.type; }
        virtual ~Sensor();
        virtual void removeSession(Session *session);
        virtual bool startStreaming(Session * session, int64_t samplingPeriodUs, int64_t maxReportLatencyUs);
        virtual bool stopStreaming(Session * session);
        virtual bool flushStreaming(Session * session);
        virtual bool setProperty(Session * session, void * property, size_t size) = 0;
        virtual size_t getProperty(Session * session, void * command, size_t commandSize, void **property) = 0;
        virtual size_t convertToEvent(const char * buf, size_t bufSize) = 0;
        virtual size_t convertToStreaming(const char * buf, size_t bufSize) = 0;
        virtual void handleEvents(int fd) = 0;
        virtual void handleEvents(const char * buf, size_t bufSize) = 0;
        virtual void setHandle(int handle) { device.handle = handle; }
        const struct sensor_device_t &getDevice() const { return device; }
};
#endif
