/*
 * Copyright (c) 2015 Intel Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "SessionList.h"
#include "Log.h"
#include <string.h>

SINGLETON_STATIC_INSTANCE(SessionList)

SessionList::SessionList()
{
        int error = pthread_mutex_init(&mutex, NULL);
        if (error != 0) {
                ALOGE("%s line: %d pthread_mutex_init error: %s", __FUNCTION__, __LINE__, strerror(error));
        }
}

SessionList::~SessionList()
{
        int error = pthread_mutex_destroy(&mutex);
        if (error != 0) {
                ALOGE("%s line: %d pthread_mutex_destroy error: %s", __FUNCTION__, __LINE__, strerror(error));
        }
}

Session * SessionList::createSession(int dataFd)
{
        Session * session = new Session(dataFd);
        mSessionList.push_back(session);

        return session;
}

bool SessionList::sessionExists(Session * session)
{
        for (std::vector<Session *>::iterator it = mSessionList.begin(); it != mSessionList.end(); it++) {
                if (*it == session)
                        return true;
        }
        return false;
}

bool SessionList::sessionComplete(Session * session)
{
        bool exists = sessionExists(session);
        if (!exists)
                return false;
        int ctrlFd = session->getControlFd();
        if (mSessionTable.count(ctrlFd))
                return true;

        return false;
}

void SessionList::updateSession(Session * session)
{
        int ctrlFd = session->getControlFd();

        if (ctrlFd < 0)
                return;

        mSessionTable[ctrlFd] = session;
}

Session * SessionList::getSession(int ctrlFd)
{
        if (mSessionTable.count(ctrlFd))
                return mSessionTable[ctrlFd];
        return NULL;
}

bool SessionList::removeSession(Session * session)
{
        if (session == NULL)
                return false;

        int error = lockSessions();
        if (error != 0) {
                ALOGE("%s line: %d lockSessions error!", __FUNCTION__, __LINE__);
                return false;
        }

        bool result = false;
        int ctrlFd = session->getControlFd();
        for (std::vector<Session *>::iterator it = mSessionList.begin(); it != mSessionList.end(); it++) {
                if (*it == session) {
                        mSessionList.erase(it);
                        result = true;
                        break;
                }
        }

        if (mSessionTable.count(ctrlFd)) {
                mSessionTable.erase(ctrlFd);
                result = true;
        }

        delete session;

        error = unlockSessions();
        if (error != 0) {
                ALOGE("%s line: %d unlockSessions error!", __FUNCTION__, __LINE__);
        }

        return result;
}

bool SessionList::removeSession(int ctrlFd)
{
        if (!mSessionTable.count(ctrlFd))
                return false;

        int error = lockSessions();
        if (error != 0) {
                ALOGE("%s line: %d lockSessions error!", __FUNCTION__, __LINE__);
                return false;
        }

        Session * session = mSessionTable[ctrlFd];
        mSessionTable.erase(ctrlFd);
        for (std::vector<Session *>::iterator it = mSessionList.begin(); it != mSessionList.end(); it++) {
                if (*it == session) {
                        mSessionList.erase(it);
                        break;
                }
        }

        delete session;

        error = unlockSessions();
        if (error != 0) {
                ALOGE("%s line: %d unlockSessions error!", __FUNCTION__, __LINE__);
        }

        return true;
}

int SessionList::lockSessions()
{
        int error = pthread_mutex_lock(&mutex);
        if (error != 0) {
                ALOGE("%s line: %d pthread_mutex_lock error: %s", __FUNCTION__, __LINE__, strerror(error));
        }

        return error;
}

int SessionList::unlockSessions()
{
        int error = pthread_mutex_unlock(&mutex);
        if (error != 0) {
                ALOGE("%s line: %d pthread_mutex_unlock error: %s", __FUNCTION__, __LINE__, strerror(error));
        }

        return error;
}
