/*
 * Copyright (c) 2015 Intel Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "Sensor.h"
#include <string.h>
#include "Log.h"
#include "SessionList.h"

static SessionList &mSessionList(SessionList::getInstance());

Sensor::Sensor(const struct sensor_device_t &device, const sensor_additional_information_t &information)
{
        memcpy(&this->device, &device, sizeof(device));
        this->information = information;

        mEvents = NULL;
        mStreaming = NULL;
        mEventsCount = 0;
        mStreamingSize = 0;
        int error = pthread_mutex_init(&batchParamsLock, NULL);
        if (error != 0) {
                ALOGE("%s line: %d pthread_mutex_init error: %s", __FUNCTION__, __LINE__, strerror(error));
        }

}

Sensor::Sensor(const Sensor * sensor)
{
        memcpy(&device, &sensor->device, sizeof(device));
        device.flags ^= SENSOR_FLAG_WAKE_UP;
        information = sensor->information;
        mEvents = NULL;
        mStreaming = NULL;
        mEventsCount = 0;
        mStreamingSize = 0;
        int error = pthread_mutex_init(&batchParamsLock, NULL);
        if (error != 0) {
                ALOGE("%s line: %d pthread_mutex_init error: %s", __FUNCTION__, __LINE__, strerror(error));
        }
}

Sensor::~Sensor()
{
        if (mEvents != NULL)
                delete [] mEvents;
        if (mStreaming != NULL)
                delete [] mStreaming;
        int error = pthread_mutex_destroy(&batchParamsLock);
        if (error != 0) {
                ALOGE("%s line: %d pthread_mutex_destroy error: %s", __FUNCTION__, __LINE__, strerror(error));
        }
}

void Sensor::removeSession(Session *session)
{
        // lockbatchParams is not necessary for reading only it only changed in this thread
        if (batchParams.count(session) > 0) {
                stopStreaming(session);
        }
        // for sensors arbitrate properties, overload this method and remove all related with this session
}

bool Sensor::startStreaming(Session * session, int64_t samplingPeriodUs, int64_t maxReportLatencyUs)
{
        if (session->sensorIsActivated(device.handle))
                session->activateSensor(device.handle, this, true);

        bool result = setBatchParamsForIdent(session, samplingPeriodUs, maxReportLatencyUs);
        selectBatchParams();

        return result;
}

bool Sensor::stopStreaming(Session * session)
{
        size_t count = removeBatchParamsForIdent(session);
        selectBatchParams();
        return count > 0 ? true : false;
}

bool Sensor::flushStreaming(Session * session)
{
        if (!batchParams.count(session)) {
                ALOGE("%s line: %d sensor %s not activated!", __FUNCTION__, __LINE__, device.name);
                return false;
        }
        return true;
}

// The same as Android Sensor Service, currently flags is ignored.
bool Sensor::setBatchParamsForIdent(Session * session, int64_t samplingPeriodNs, int64_t maxBatchReportLatencyNs)
{
        if (batchParams.count(session)) {
                BatchParams& params = batchParams[session];
                params.batchDelay = samplingPeriodNs;
                params.batchTimeout = maxBatchReportLatencyNs;
        } else {
                BatchParams params;
                params.batchDelay = samplingPeriodNs;
                params.batchTimeout = maxBatchReportLatencyNs;
                lockbatchParams();
                batchParams[session] = params;
                unlockbatchParams();
        }

        return true;
}

void Sensor::selectBatchParams()
{
        BatchParams bestParams(-1, -1);

        if (batchParams.size() > 0) {
                std::map<Session *, BatchParams>::iterator it = batchParams.begin();
                bestParams = it->second;
                it++;
                for (; it != batchParams.end(); it++) {
                        BatchParams params = it->second;
                        if (params.batchDelay < bestParams.batchDelay) {
                                bestParams.batchDelay = params.batchDelay;
                        }
                        if (params.batchTimeout < bestParams.batchTimeout) {
                                bestParams.batchTimeout = params.batchTimeout;
                        }
                }
        }
        bestBatchParams = bestParams;
}

size_t Sensor::removeBatchParamsForIdent(Session * session)
{
        lockbatchParams();
        size_t count = batchParams.erase(session);
        unlockbatchParams();
        if (count > 0) {
                selectBatchParams();
        }
        return count;
}

void Sensor::reallocStreamingBuffer(size_t size)
{
        if (mStreamingSize < size) {
                if (mStreaming != NULL)
                        delete [] mStreaming;
                mStreaming = new char[size];
                mStreamingSize = size;
        }
}

void Sensor::reallocEventsBuffer(size_t count)
{
        if (mEventsCount < count) {
                if (mEvents != NULL)
                        delete [] mEvents;
                mEvents = new struct sensors_event_t[count];
                mEventsCount = count;
                memset(mEvents, 0, count * sizeof(struct sensors_event_t));
                for (size_t i = 0; i < count; i++) {
                        mEvents[i].version = sizeof(struct sensors_event_t);
                        mEvents[i].sensor = device.handle;
                        mEvents[i].type = device.type;
                }
        }
}

int Sensor::lockbatchParams()
{
        int error = pthread_mutex_lock(&batchParamsLock);
        if (error != 0) {
                ALOGE("%s line: %d pthread_mutex_lock error: %s", __FUNCTION__, __LINE__, strerror(error));
        }

        return error;
}

int Sensor::unlockbatchParams()
{
        int error = pthread_mutex_unlock(&batchParamsLock);
        if (error != 0) {
                ALOGE("%s line: %d pthread_mutex_unlock error: %s", __FUNCTION__, __LINE__, strerror(error));
        }

        return error;
}
