/******************************************************************************
 * Copyright (C) 2003-2004 by                  NTRU Cryptosystems, Inc.       *
 *                         ---ALL RIGHTS RESERVED---                          *
 *                                                                            *
 * This software is proprietary and confidential to NTRU Cryptosystems, Inc., *
 * is furnished under a license and may be used and copied only in accordance *
 * with the terms of such license and with the inclusion of the above         *
 * copyright notice. This software or any other copies thereof may not be     *
 * provided or otherwise made available to any other person. No title to and  *
 * ownership of the software is hereby transferred.                           *
 *                                                                            *
 * The information in this software is subject to change without notice       *
 * and should not be construed as a commitment by NTRU Cryptosystems, Inc.    *
 *                                                                            *
 ******************************************************************************
 *
 * Contents: The header file for the OTP TPM functionality.
 *   This includes the TPM OTP spec v0.19, and the TSS spec v??
 *
 *****************************************************************************/

#if !defined(_TSS_OTP_H_)
#define _TSS_OTP_H_

#include "tspi.h"

#if defined ( __cplusplus )
extern "C" {
#endif /* __cplusplus */

//////////////////////////////////////////////////////////////////////////
// Misc Definitions
//////////////////////////////////////////////////////////////////////////

#define AES_BLOCKSIZE 16


//////////////////////////////////////////////////////////////////////////
// TSS Definitions
//////////////////////////////////////////////////////////////////////////

//
// Definition of the object types that can be created via CreateObject
//
#define   TSS_OBJECT_TYPE_OTP    (0xFD)      // OTP object

//
// Attributes of the OTP_TOKEN object type.
//
#define TSS_TSPATTRIB_OTP_TOKEN_PROVISIONING                (0x00000000)
#define TSS_TSPATTRIB_OTP_TOKEN_ID                          (0x00000001)
#define TSS_TSPATTRIB_OTP_ID_DIGEST                         (0x00000003)
#define TSS_TSPATTRIB_OTP_TOKEN_ALG                         (0x00000004)
#define TSS_TSPATTRIB_OTP_FIRMWARE_VERSION                  (0x00000005)
#define TSS_TSPATTRIB_OTP_TOKEN_BLOB_REVISION               (0x00000006)
#define TSS_TSPATTRIB_OTP_RSA_PARMS                         (0x00000007)
#define TSS_TSPATTRIB_OTP_ENCDATA                           (0x00000008)
#define TSS_TSPATTRIB_OTP_TOKEN_BLOB                        (0x00000009)

// Attribute subflags.
#define TSS_TSPATTRIB_OTP_RSAPARMS_SERIAL_NUMBER            (0x00000000)
#define TSS_TSPATTRIB_OTP_RSAPARMS_DISPLAY_DIGITS           (0x00000001)
#define TSS_TSPATTRIB_OTP_RSAPARMS_DISPLAY_INTERVAL_SECS    (0x00000002)

//
// Values for the OTP TSS_TSPATTRIB_OTP_TOKEN_PROVISIONING attribute.
//
#define TSS_OTP_TOKENPROV_NORMAL                            (0x00000001)
#define TSS_OTP_TOKENPROV_DIRECT                            (0x00000002)
#define TSS_OTP_TOKENPROV_DIRECT_ENC                        (0x00000003)

//
// Values for the OTP TSS_TSPATTRIB_OTP_TOKEN_ALG attribute.
//
#define TSS_OTP_TOKENALG_OATH                               (0x00000001)
#define TSS_OTP_TOKENALG_RSA                                (0x00000002)

//
// Values for the TSS_OTP_MAC_TYPE used by Tspi_OTP_ComputeMAC.
//
#define TSS_OTP_MACTYPE_MAC1                                (0x00000001)
#define TSS_OTP_MACTYPE_MAC2                                (0x00000002)


//////////////////////////////////////////////////////////////////////////
// Return codes
//////////////////////////////////////////////////////////////////////////

//
// MessageId: TPM_E_OTP_INVALID_TOKEN_DIGEST
//
// MessageText:
//
// The TPM_OTP_TOKEN_ENC blob did not verify.
//
#define TPM_E_OTP_INVALID_TOKEN_DIGEST ((UINT32)(TPM_E_BASE + 0x0000042a))

//
// MessageId: TPM_E_OTP_INVALID_TOKEN_TPMPROOF
//
// MessageText:
//
// Invalid TPM Proof in TPM_OTP_TOKEN_ENC structure.
//
#define TPM_E_OTP_INVALID_TOKEN_TPMPROOF ((UINT32)(TPM_E_BASE + 0x0000042b))

//
// MessageId: TPM_E_OTP_TOKEN_NOT_ACTIVATED
//
// MessageText:
//
// The toekn has not been activated.
//
#define TPM_E_OTP_TOKEN_NOT_ACTIVATED ((UINT32)(TPM_E_BASE + 0x0000042c))

//
// MessageId: TPM_E_OTP_NOT_ENABLED
//
// MessageText:
//
// The OTP functionality has not been enabled.
//
#define TPM_E_OTP_NOT_ENABLED ((UINT32)(TPM_E_BASE + 0x0000042d))

//
// MessageId: TPM_E_OTP_SHARED_SECRET_GENERATION_FAILED
//
// MessageText:
//
// The generation of a shared secret failed.
//
#define TPM_E_OTP_SHARED_SECRET_GENERATION_FAILED ((UINT32)(TPM_E_BASE + 0x0000042e))

//
// MessageId: TSS_E_OTP_NO_TOKEN
//
// MessageText:
//
// The OTP Token object does not have a blob and the operation requires one.
//
#define TSS_E_OTP_NO_TOKEN                        ((UINT32)(TPM_E_BASE + 0x00000440))

//////////////////////////////////////////////////////////////////////////
// Derived types
//////////////////////////////////////////////////////////////////////////

typedef UINT32      TSS_OTP_TOKEN_PROV;
typedef UINT32      TSS_OTP_TOKEN_ID;
typedef UINT32      TSS_OTP_TOKEN_ALG;
typedef UINT32      TSS_OTP_MAC_TYPE;

typedef TSS_HOBJECT TSS_HTOKEN_OTP;

//////////////////////////////////////////////////////////////////////////
// Structures
//////////////////////////////////////////////////////////////////////////

typedef struct tdTSS_BRCM_CAPS
{
    TSS_VERSION versionInfo;
    UINT32 prodConfigReg;
    UINT32 OTPFirmwareVer;
} TSS_BRCM_CAPS;

typedef struct tdTSS_OTP_ISO_TIME
{
    TSS_VERSION versionInfo;
    BYTE        time[7];
} TSS_OTP_ISO_TIME;

//////////////////////////////////////////////////////////////////////////
// Tspi function declarations
//////////////////////////////////////////////////////////////////////////

TSPICALL Tspi_TPM_EnableOTP
(
    TSS_HTPM            hTPM,                           // in
    TSS_BOOL            bEnable                         // in
);

TSPICALL Tspi_OTP_CreateToken
(
    TSS_HTOKEN_OTP      hOTPToken,                      // in
    TSS_HKEY            hTokenParentKey,                // in
    TSS_HKEY            hServerPubKey,                  // in
    UINT32*             encNonceSize,                   // out
    BYTE**              encNonce                        // out
);

TSPICALL Tspi_OTP_ComputeMAC
(
    TSS_HTOKEN_OTP      hOTPToken,                      // in
    TSS_HKEY            hTokenParentKey,                // in
    TSS_OTP_MAC_TYPE    macType,                        // in
    UINT32              serverRandomSize,               // in
    BYTE*               serverRandom,                   // in
    UINT32*             macSize,                        // out
    BYTE**              mac                             // out
);

TSPICALL Tspi_OTP_ActivateToken
(
    TSS_HTOKEN_OTP      hOTPToken,                      // in
    TSS_HKEY            hTokenParentKey,                // in
    TSS_HKEY            hServerPubKey,                  // in
    UINT32              sharedSecretSeedSize,           // in
    BYTE*               sharedSecretSeed                // in
);

TSPICALL Tspi_OTP_Authenticate
(
    TSS_HTOKEN_OTP      hOTPToken,                      // in
    TSS_HKEY            hTokenParentKey,                // in
    UINT32              otpParmsSize,                   // in
    BYTE*               otpParms,                       // in
    UINT32*             otpValueSize,                   // out
    BYTE**              otpValue                        // out
);

TSPICALL Tspi_OTP_CertifyToken
(
    TSS_HTOKEN_OTP      hOTPToken,                      // in
    TSS_HKEY            hTokenParentKey,                // in
    TSS_HKEY            hCertKey,                       // in
    UINT32              otpParmsSize,                   // in
    BYTE*               otpParms,                       // in
    TSS_VALIDATION     *validation,                     // in,out
    UINT32*             otpValueSize,                   // out
    BYTE**              otpValue                        // out
);





//////////////////////////////////////////////////////////////////////////
// TPM Definitions
//////////////////////////////////////////////////////////////////////////

#define TPM_ORD_vendorOTPEnableOTP          0x2000001d
#define TPM_ORD_vendorOTPCreateToken        0x2000001e
#define TPM_ORD_vendorOTPActivateToken      0x2000001f
#define TPM_ORD_vendorOTPAuthenticate       0x20000020
#define TPM_ORD_vendorOTPCertifyToken       0x20000021
#define TPM_ORD_vendorOTPComputeMAC         0x20000023

// Structure tags
#define TPM_TAG_OTP_RSA_ALG_PARMS           0x2001
#define TPM_TAG_OTP_TOKEN                   0x2002
#define TPM_TAG_OTP_CERTIFY_TOKEN_INFO      0x2003

typedef UINT32 TPM_OTP_TOKEN_PROV;
#define TPM_OTP_TOKENPROV_NORMAL      1
#define TPM_OTP_TOKENPROV_DIRECT      2
#define TPM_OTP_TOKENPROV_DIRECT_ENC  3

typedef UINT32 TPM_OTP_TOKEN_ID;

typedef UINT32 TPM_OTP_TOKEN_ALG;
#define TPM_OTP_ALG_OATH  1
#define TPM_OTP_ALG_RSA   2

typedef UINT32 TPM_OTP_MAC_TYPE;
#define TPM_OTP_MAC1    1
#define TPM_OTP_MAC2    2


typedef struct tdTPM_BRCM_CAPS {
    UINT32            prodConfigReg;
    UINT32            OTPFirmwareVer;
} TPM_BRCM_CAPS;

typedef struct tdTPM_OTP_NONCE
{
    BYTE nonce[AES_BLOCKSIZE];
} TPM_OTP_NONCE;

typedef struct tdTPM_OTP_RSA_MASTER_SEED
{
    BYTE seed[AES_BLOCKSIZE];
} TPM_OTP_RSA_MASTER_SEED;

#define TPM_OTP_ISO_TIME_SIZE  7
typedef struct tdTPM_OTP_ISO_TIME
{
    BYTE time[TPM_OTP_ISO_TIME_SIZE];
} TPM_OTP_ISO_TIME;

typedef struct tdTPM_OTP_RSA_SERIAL_NUMBER
{
    BYTE number[4];
} TPM_OTP_RSA_SERIAL_NUMBER;

typedef struct tdTPM_OTP_RSA_ALG_PARMS
{
    TPM_STRUCTURE_TAG         tag;
    TPM_OTP_RSA_SERIAL_NUMBER serialNumber;
    BYTE                      displayDigits;
    BYTE                      displayIntervalSeconds;
} TPM_OTP_RSA_ALG_PARMS;


typedef struct tdTPM_OTP_TOKEN 
{
    TPM_STRUCTURE_TAG  tag;
    TPM_OTP_TOKEN_PROV tokenProvisioning;
    TPM_OTP_TOKEN_ID   tokenID;
    TPM_DIGEST         IDDigest;
    TPM_OTP_TOKEN_ALG  tokenAlg;
    UINT32             OTPFirmwareVer;
    UINT32             tokenBlobRevision;
    UINT32             algParmsSize;
    BYTE              *algParms;
    UINT32             encDataSize;
    BYTE              *encData;
} TPM_OTP_TOKEN;

typedef struct tdTPM_OTP_CERTIFY_TOKEN_INFO
{
    TPM_STRUCTURE_TAG  tag;
    TPM_OTP_TOKEN_PROV tokenProvisioning;
    TPM_OTP_TOKEN_ID   tokenID;
    TPM_OTP_TOKEN_ALG  tokenAlg;
    UINT32             OTPFirmwareVer;
    UINT32             serialNumberSize;
    TPM_OTP_RSA_SERIAL_NUMBER serialNumber;
    UINT32             OTPValueSize;
    BYTE              *OTPValue;
} TPM_OTP_CERTIFY_TOKEN_INFO;


#if defined ( __cplusplus )
}
#endif /* __cplusplus */

#endif /* _TSS_OTP_H_ */
