#
# Author: Guillermo Gonzalez <guillermo.gonzalez@canonical.com>
#
# Copyright 2009 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
""" Base tests cases and test utilities """
from __future__ import with_statement

import dbus
from dbus.mainloop.glib import DBusGMainLoop
import logging
import os
import shutil

from ubuntuone.oauthdesktop.main import Login, LoginProcessor
from ubuntuone.syncdaemon import (
    config,
    action_queue,
    event_queue,
    filesystem_manager as fs_manager,
    interfaces,
    volume_manager,
    main,
)
from ubuntuone.syncdaemon.dbus_interface import (
    DBusInterface,
    DBusExposedObject,
    NM_STATE_CONNECTED,
    NM_STATE_DISCONNECTED,
    DBUS_IFACE_AUTH_NAME,
)
from twisted.internet import defer
from twisted.trial.unittest import TestCase as TwistedTestCase
from zope.interface import implements
from oauth import oauth

class FakeOAuthClient(object):
    """ Fake OAuthClient"""

    def __init__(self, realm):
        """ create the instance. """
        self.realm = realm
        self.consumer = oauth.OAuthConsumer('ubuntuone', 'hammertime')

    def get_access_token(self):
        """ returns a Token"""
        return 'a token'


class FakeHashQueue(object):
    """A fake hash queue"""
    def empty(self):
        """are we empty? sure we are"""
        return True

    def shutdown(self):
        """go away? I'l barely *here*!"""
        pass

    def __len__(self):
        """ length is 0. we are empty, right?"""
        return 0


class FakeMark(object):
    """A fake Mark Shuttleworth..."""
    def stop(self):
        """...that only knows how to stop"""


class FakeDBusInterface(object):
    """A fake DBusInterface..."""
    def shutdown(self, with_restart=False):
        """...that only knows how to go away"""


class FakeMain(main.Main):
    """ A fake Main class to setup the tests """

    # don't call Main.__init__ we take care of creating a fake main and
    # all its attributes. pylint: disable-msg=W0231
    def __init__(self, root_dir, shares_dir, data_dir, partials_dir):
        """ create the instance. """
        self.logger = logging.getLogger('ubuntuone.SyncDaemon.FakeMain')
        self.root_dir = root_dir
        self.data_dir = data_dir
        self.shares_dir = shares_dir
        self.partials_dir = partials_dir
        self.shares_dir_link = os.path.join(self.root_dir, 'Shared With Me')
        self.realm = 'http://test.ubuntuone.com'
        self.oauth_client = FakeOAuthClient(self.realm)
        self.vm = volume_manager.VolumeManager(self)
        self.fs = fs_manager.FileSystemManager(self.data_dir,
                                               self.partials_dir, self.vm)
        self.event_q = event_queue.EventQueue(self.fs)
        self.fs.register_eq(self.event_q)
        self.action_q = FakeActionQueue(self.event_q)
        self.state = main.SyncDaemonStateManager(self, 2, 0)
        self.event_q.subscribe(self.vm)
        self.vm.init_root()
        self.hash_q = FakeHashQueue()
        self.mark = FakeMark()
        self.dbus_iface = FakeDBusInterface()

    def _connect_aq(self, token):
        """ Connect the fake action queue """
        self.action_q.connect()

    def _disconnect_aq(self):
        """ Disconnect the fake action queue """
        self.action_q.disconnect()

    def get_access_token(self):
        """fake get token"""
        return None

    def check_version(self):
        """ Check the client protocol version matches that of the server. """
        self.event_q.push('SYS_PROTOCOL_VERSION_OK')

    def authenticate(self):
        """ Do the OAuth dance. """
        self.event_q.push('SYS_OAUTH_OK')

    def set_capabilities(self):
        """Set the capabilities"""
        self.event_q.push('SYS_SET_CAPABILITIES_OK')

    def get_root(self, root_mdid):
        """ Ask que AQ for our root's uuid. """
        return defer.succeed('root_uuid')

    def server_rescan(self):
        """ Do the server rescan? naaa! """
        self.event_q.push('SYS_SERVER_RESCAN_STARTING')
        self.event_q.push('SYS_SERVER_RESCAN_DONE')
        return defer.succeed('root_uuid')



class BaseTwistedTestCase(TwistedTestCase):
    """ Base TestCase that provides:
        mktemp(name): helper to create temporary dirs
    """

    def mktemp(self, name='temp'):
        """ Customized mktemp that accepts an optional name argument. """
        tempdir = os.path.join(self.tmpdir, name)
        if os.path.exists(tempdir):
            self.rmtree(tempdir)
        self.makedirs(tempdir)
        return tempdir

    @property
    def tmpdir(self):
        """ default tmpdir: 'module name'/'class name'"""
        MAX_FILENAME = 32 # some platforms limit lengths of filenames
        base = os.path.join(self.__class__.__module__[:MAX_FILENAME],
                            self.__class__.__name__[:MAX_FILENAME],
                            self._testMethodName[:MAX_FILENAME])
        # use _trial_temp dir, it should be os.gwtcwd()
        # define the root temp dir of the testcase, pylint: disable-msg=W0201
        self.__root = os.path.join(os.getcwd(), base)
        return os.path.join(self.__root, 'tmpdir')

    def rmtree(self, path):
        """ rmtree that handle ro parent(s) and childs. """
        # change perms to rw, so we can delete the temp dir
        if path != getattr(self, '__root', None):
            os.chmod(os.path.dirname(path), 0755)
        os.chmod(path, 0755)
        # pylint: disable-msg=W0612
        for dirpath, dirs, files in os.walk(path):
            for dir in dirs:
                os.chmod(os.path.join(dirpath, dir), 0777)
        shutil.rmtree(path)

    def makedirs(self, path):
        """ makedirs that handle ro parent. """
        parent = os.path.dirname(path)
        if os.path.exists(parent):
            os.chmod(parent, 0755)
        os.makedirs(path)

    def setUp(self):
        TwistedTestCase.setUp(self)
        # invalidate the current config
        config_file = os.path.join(self.mktemp('config'), 'syncdaemon.conf')
        # fake a very basic config file with sane defaults for the tests
        with open(config_file, 'w') as fp:
            fp.write('[bandwidth_throttling]\n')
            fp.write('on = False\n')
            fp.write('read_limit = -1\n')
            fp.write('write_limit = -1\n')
        config._user_config = None
        config.get_user_config(config_file=config_file)

    def tearDown(self):
        """ cleanup the temp dir. """
        # invalidate the current config
        config._user_config = None
        config.get_user_config()
        if hasattr(self, '__root'):
            self.rmtree(self.__root)
        return TwistedTestCase.tearDown(self)


class DBusTwistedTestCase(BaseTwistedTestCase):
    """ Test the DBus event handling """

    def setUp(self):
        """ Setup the infrastructure fo the test (dbus service). """
        BaseTwistedTestCase.setUp(self)
        self.log = logging.getLogger("ubuntuone.SyncDaemon.TEST")
        self.log.info("starting test %s.%s", self.__class__.__name__,
                      self._testMethodName)
        self.timeout = 2
        self.data_dir = self.mktemp('data_dir')
        self.partials_dir = self.mktemp('partials')
        self.root_dir = self.mktemp('root_dir')
        self.shares_dir = self.mktemp('shares_dir')
        self.main = FakeMain(self.root_dir, self.shares_dir,
                             self.data_dir, self.partials_dir)
        self.fs_manager = self.main.fs
        self.event_q = self.main.event_q
        self.action_q = self.main.action_q
        self.loop = DBusGMainLoop(set_as_default=True)
        self.bus = dbus.bus.BusConnection(mainloop=self.loop)
        self.nm = FakeNetworkManager(self.bus)
        # monkeypatch busName.__del__ to avoid errors on gc
        # we take care of releasing the name in shutdown
        dbus.service.BusName.__del__ = lambda _: None
        self.dbus_iface = DBusInterface(self.bus, self.main,
                                        system_bus=self.bus)
        self.main.dbus_iface = self.dbus_iface
        self.busName = self.dbus_iface.busName
        self.bus.set_exit_on_disconnect(False)
        self.dbus_iface.connect()
        self.event_q.push('SYS_WAIT_FOR_LOCAL_RESCAN')
        self.event_q.push('SYS_LOCAL_RESCAN_DONE')
        self.signal_receivers = set()
        self.action_q.content_queue.set_change_notification_cb(
            self.dbus_iface.status.emit_content_queue_changed)

    def tearDown(self):
        """ Cleanup the test. """
        d = self.cleanup_signal_receivers(self.signal_receivers)
        d.addBoth(self._tearDown)
        d.addBoth(lambda _: BaseTwistedTestCase.tearDown(self))
        return d

    def _tearDown(self, *args):
        """ shutdown """
        self.main.shutdown()
        self.nm.shutdown()
        self.dbus_iface.bus.flush()
        self.bus.flush()
        self.bus.close()
        self.rmtree(self.shares_dir)
        self.rmtree(self.root_dir)
        self.rmtree(self.data_dir)
        self.rmtree(self.partials_dir)
        self.log.info("finished test %s.%s", self.__class__.__name__,
                      self._testMethodName)

    def error_handler(self, *error):
        """ default error handler for DBus calls. """
        self.fail(error)

    def cleanup_signal_receivers(self, signal_receivers):
        """ cleanup self.signal_receivers and returns a deferred """
        deferreds = []
        for match in signal_receivers:
            d = defer.Deferred()
            def callback(*args):
                """ callback that accepts *args. """
                d.callback(args)
            self.bus.call_async(dbus.bus.BUS_DAEMON_NAME,
                                dbus.bus.BUS_DAEMON_PATH,
                                dbus.bus.BUS_DAEMON_IFACE, 'RemoveMatch', 's',
                                (str(match),), callback, self.error_handler)
            deferreds.append(d)
        if deferreds:
            return defer.DeferredList(deferreds)
        else:
            return defer.succeed(True)



class FakeActionQueue(object):
    """ stub implementation """

    implements(interfaces.IActionQueue)

    def __init__(self, eq, *args, **kwargs):
        """ Creates the instance """
        self.eq = self.event_queue = eq
        self.client = action_queue.ActionQueueProtocol()
        self.client.disconnect = lambda: None
        self.uploading = {}
        self.downloading = {}
        # pylint: disable-msg=C0103
        class UUID_Map(object):
            """mock uuid map"""
            def set(self, *args):
                """mock set method"""
                pass

        self.uuid_map = UUID_Map()
        self.content_queue = action_queue.ContentQueue('CONTENT_QUEUE', self)
        self.meta_queue = action_queue.RequestQueue('META_QUEUE', self)

        # throttling attributes
        self.readLimit = None
        self.writeLimit = None
        self.throttling = False

    def connect(self, host=None, port=None, user_ssl=False):
        """ stub implementation """
        self.eq.push('SYS_CONNECTION_MADE')

    def disconnect(self):
        """ stub implementation """
        pass

    def enable_throttling(self, value):
        self.throttling = value

    def cancel_download(self, share_id, node_id):
        """ stub implementation """
        pass

    def cancel_upload(self, share_id, node_id):
        """ stub implementation """
        pass

    def download(self, share_id, node_id, server_hash, fileobj):
        """ stub implementation """
        pass

    def upload(self, share_id, node_id, previous_hash, hash, crc32,
               size, deflated_size, fileobj):
        """ stub implementation """
        pass

    def make_file(self, share_id, parent_id, name, marker):
        """ stub implementation """
        pass

    def make_dir(self, share_id, parent_id, name, marker):
        """ stub implementation """
        pass

    def move(self, share_id, node_id, old_parent_id, new_parent_id, new_name):
        """ stub implementation """
        pass

    def unlink(self, share_id, node_id):
        """ stub implementation """
        pass

    def query(self, items):
        """ stub implementation """
        pass

    def listdir(self, share_id, node_id, server_hash, fileobj):
        """ stub implementation """
        pass

    def list_shares(self):
        """ stub implementation """
        pass

    def answer_share(self, share_id, answer):
        """ stub implementation """
        self.eq.push('AQ_ANSWER_SHARE_OK', share_id, answer)

    def create_share(self, *args):
        """ sutb implementation """
        pass

    def inquire_free_space(self, share_id):
        """ stub implementation """
        pass

    def inquire_account_info(self):
        """ stub implementation """
        pass


class MementoHandler(logging.Handler):
    """ A handler class which store logging records in a list """

    def __init__(self, *args, **kwargs):
        """ Create the instance, and add a records attribute. """
        logging.Handler.__init__(self, *args, **kwargs)
        self.records = []

    def emit(self, record):
        """ Just add the record to self.records. """
        self.records.append(record)


class FakeNetworkManager(DBusExposedObject):
    """ A fake NetworkManager that only emits StatusChanged signal. """

    State = 3

    def __init__(self, bus):
        """ Creates the instance. """
        self.path = '/org/freedesktop/NetworkManager'
        self.bus = bus
        self.bus.request_name('org.freedesktop.NetworkManager',
                              flags=dbus.bus.NAME_FLAG_REPLACE_EXISTING | \
                              dbus.bus.NAME_FLAG_DO_NOT_QUEUE | \
                              dbus.bus.NAME_FLAG_ALLOW_REPLACEMENT)
        self.busName = dbus.service.BusName('org.freedesktop.NetworkManager',
                                        bus=self.bus)
        DBusExposedObject.__init__(self, bus_name=self.busName,
                                   path=self.path)

    def shutdown(self):
        """ Shutdown the fake NetworkManager """
        self.busName.get_bus().release_name(self.busName.get_name())
        self.remove_from_connection()

    @dbus.service.signal('org.freedesktop.NetworkManager', signature='i')
    # pylint: disable-msg=C0103
    def StateChanged(self, state):
        """ Fire DBus signal StatusChanged. """
        pass

    def emit_connected(self):
        """ Emits the signal StateCganged(3). """
        self.StateChanged(NM_STATE_CONNECTED)

    def emit_disconnected(self):
        """ Emits the signal StateCganged(4). """
        self.StateChanged(NM_STATE_DISCONNECTED)

    @dbus.service.method(dbus.PROPERTIES_IFACE,
                         in_signature='ss', out_signature='v',
                         async_callbacks=('reply_handler', 'error_handler'))
    # pylint: disable-msg=C0103
    def Get(self, interface, propname, reply_handler=None, error_handler=None):
        """
        Fake dbus's Get method to get at the State property
        """
        try:
            reply_handler(getattr(self, propname, None))
        except Exception, e: # pylint: disable-msg=W0703
            error_handler(e)

class FakeVolumeManager(object):
    """ A volume manager that only knows one share, the root"""

    def __init__(self, root_path):
        """ Creates the instance"""
        self.root = volume_manager.Share(root_path, access_level='Modify')
        self.shares = {'':self.root}
        self.log = logging.getLogger('ubuntuone.SyncDaemon.VM-test')

    def add_share(self, share):
        """ Adss share to the shares dict """
        self.shares[share.id] = share
        # if the share don't exists, create it
        if not os.path.exists(share.path):
            os.mkdir(share.path)
        # if it's a ro share, change the perms
        if not share.can_write():
            os.chmod(share.path, 0555)

    def on_server_root(self, _):
        """
        Do nothing
        """

# OAuth stubs
class FakeLoginProcessor(LoginProcessor):
    """Stub login processor."""

    def __init__(self, dbus_object):
        """Initialize the login processor."""
        LoginProcessor.__init__(self, dbus_object, use_libnotify=False)
        self.next_login_cb = None

    def login(self, realm, consumer_key, error_handler=None, reply_handler=None, do_login=True):
        """Stub, call self.next_login_cb or send NewCredentials if
        self.next_login_cb isn't defined.
        """
        self.realm = str(realm)
        self.consumer_key = str(consumer_key)
        if self.next_login_cb:
            cb = self.next_login_cb[0]
            args = self.next_login_cb[1]
            self.next_login_cb = None
            return cb(*args)
        else:
            self.dbus_object.NewCredentials(realm, consumer_key)

    def clear_token(self, realm, consumer_key):
        """Stub, do nothing"""
        pass

    def next_login_with(self, callback, args=tuple()):
        """shortcircuit the next call to login and call the specified callback.
        callback is usually one of: self.got_token, self.got_no_token,
        self.got_denial or self.got_error.
        """
        self.next_login_cb = (callback, args)


class FakeLogin(Login):
    """Stub Object which listens for D-Bus OAuth requests"""

    def __init__(self, bus):
        """Initiate the object."""
        self.bus = bus
        self.busName = dbus.service.BusName(DBUS_IFACE_AUTH_NAME, bus=self.bus)
        # bypass the parent class __init__ as it has the path hardcoded
        # and we can't use '/' as the path, as we are already using it
        # for syncdaemon. pylint: disable-msg=W0233,W0231
        dbus.service.Object.__init__(self, object_path="/oauthdesktop",
                                     bus_name=self.busName)
        self.processor = FakeLoginProcessor(self)
        self.currently_authing = False

    def shutdown(self):
        """Shutdown and remove any trace from the bus"""
        self.busName.get_bus().release_name(self.busName.get_name())
        self.remove_from_connection()

