# -*- coding: utf-8 -*-
#
# Author: Guillermo Gonzalez <guillermo.gonzalez@canonical.com>
#
# Copyright 2009 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
""" Tests for the DBUS interface """

import inspect
import os
import unittest

from ubuntuone.storageprotocol.sharersp import (
    NotifyShareHolder,
)
from ubuntuone.syncdaemon import dbus_interface
from ubuntuone.syncdaemon.dbus_interface import (
    DBUS_IFACE_STATUS_NAME,
    DBUS_IFACE_EVENTS_NAME,
    DBUS_IFACE_FS_NAME,
    DBUS_IFACE_SYNC_NAME,
    DBUS_IFACE_SHARES_NAME,
    DBUS_IFACE_CONFIG_NAME,
    EventListener,
)
from ubuntuone.syncdaemon.volume_manager import Share
from ubuntuone.syncdaemon.tools import DBusClient
from ubuntuone.syncdaemon import event_queue
from ubuntuone.syncdaemon import states, main
from contrib.testing.testcase import (
    DBusTwistedTestCase,
    FakeLogin,
)
from twisted.internet import defer
from ubuntuone.syncdaemon.marker import MDMarker


class DBusInterfaceTests(DBusTwistedTestCase):
    """ Basic tests to the objects exposed with D-Bus"""

    def test_status(self):
        """ Test the Status Object, registering it to the session bus, and
        calling current_uploads method.
        """
        client = DBusClient(self.bus, '/status', DBUS_IFACE_STATUS_NAME)
        d = defer.Deferred()
        def waiting_handler(result):
            """ waiting handler """
            self.assertEquals(result, [])
            d.callback(True)

        def status_handler(result):
            """ reply handler """
            state = states.IDLE
            self.assertEquals(state.name, result['name'])
            self.assertEquals(state.description, result['description'])
            self.assertEquals(state.is_error, bool(result['is_error']))
            self.assertEquals(state.is_connected, bool(result['is_connected']))
            self.assertEquals(state.is_online, bool(result['is_online']))
            client.call_method('waiting_content',
                               reply_handler=waiting_handler,
                               error_handler=self.error_handler)

        def downloads_handler(result):
            """ reply handler """
            self.assertEquals(0, len(result))
            client.call_method('current_status',
                               reply_handler=status_handler,
                               error_handler=self.error_handler)

        def uploads_handler(result):
            """ reply handler """
            self.assertEquals(0, len(result))
            client.call_method('current_downloads',
                               reply_handler=downloads_handler,
                               error_handler=self.error_handler)

        client.call_method('current_uploads', reply_handler=uploads_handler,
                           error_handler=self.error_handler)
        return d

    def test_syncDaemon(self):
        """ Test the SyncDaemon object, registering it to the session bus, and
        calling connect and disconnect methods.
        """
        client = DBusClient(self.bus, '/', DBUS_IFACE_SYNC_NAME)
        d = defer.Deferred()
        def disconnect_handler(result):
            """ reply handler """
            self.assertEquals(None, result)
            d.callback(True)

        def connect_handler(result):
            """ reply handler """
            self.assertEquals(None, result)
            client.call_method("disconnect", reply_handler=disconnect_handler,
                               error_handler=self.error_handler)

        client.call_method("connect", reply_handler=connect_handler,
                           error_handler=self.error_handler)

        return d

    def test_filesystem(self):
        """ Test the FileSystem Object, registering it to the session bus, and
        excercising the API.
        """
        share_path = os.path.join(self.shares_dir, 'share')
        self.main.vm.add_share(Share(share_path, share_id='share'))
        path = os.path.join(share_path, "foo")
        self.fs_manager.create(path, "share")
        self.fs_manager.set_node_id(path, "node_id")
        d = defer.Deferred()
        def handler(metadata):
            """ reply handler """
            d.callback(metadata)

        def callback(result):
            """ callback to check the result. """
            self.assertEquals('foo', str(result['path']))
            self.assertEquals('share', result['share_id'])
            self.assertEquals('node_id', result['node_id'])

        d.addCallback(callback)
        client = DBusClient(self.bus, '/filesystem', DBUS_IFACE_FS_NAME)
        client.call_method('get_metadata', path, reply_handler=handler,
                           error_handler=self.error_handler)
        return d

    def test_push_event(self):
        """ Test the exposed method: push_event """
        push_deferred = defer.Deferred()
        queue_deferred = defer.Deferred()
        push_deferred.addCallback(lambda _: queue_deferred)
        class Listener(object):
            """ A basic listener to handle the pushed event. """

            def handle_FS_FILE_CREATE(self, path):
                """ FS_FILE_CREATE handling """
                self.assertEquals('bar', path)

        listener = Listener()
        self.event_q.subscribe(listener)
        self.event_q.unsubscribe(self.dbus_iface.event_listener)

        def empty_queue_callback():
            """
            Called when the event queue empties
            """
            queue_deferred.callback(True)

        def default(path):
            """ default callback to assert the pushed event. """
            self.event_q.unsubscribe(listener)
            self.event_q.subscribe(self.dbus_iface.event_listener)
            self.event_q.remove_empty_event_queue_callback(empty_queue_callback)

        queue_deferred.addCallback(default)

        client = DBusClient(self.bus, '/events',
                   DBUS_IFACE_EVENTS_NAME)
        client.call_method(
            'push_event', 'FS_FILE_CREATE', ('bar',),
            reply_handler=lambda *_: push_deferred.callback(True),
            error_handler=self.error_handler)
        self.event_q.add_empty_event_queue_callback(empty_queue_callback)
        return push_deferred

    def test_waiting_content(self):
        """
        Test Status.waiting_content and Status.schedule_next with fake
        data in the AQ
        """
        class FakeCommand(object):
            """A fake command"""
            def __init__(self, share_id, node_id):
                """create it"""
                self.share_id = share_id
                self.node_id = node_id
            def is_runnable(self):
                '''is runnable'''
                return True

        # prepare the VM so it lies for us
        share_path = os.path.join(self.shares_dir, 'share')
        self.main.vm.add_share(Share(share_path, share_id='share_id'))
        a_path = os.path.join(share_path, "path_a")
        b_path = os.path.join(share_path, "path_b")
        self.fs_manager.create(a_path, "share_id")
        self.fs_manager.create(b_path, "share_id")
        self.fs_manager.set_node_id(a_path, "node_id_a")
        self.fs_manager.set_node_id(b_path, "node_id_b")
        # inject the fake data
        self.action_q.content_queue.waiting.extend([
                FakeCommand("share_id", "node_id_a"),
                FakeCommand("share_id", "node_id_b")])
        # OK, testing time
        client = DBusClient(self.bus, '/status', DBUS_IFACE_STATUS_NAME)
        d = defer.Deferred()
        self.second = False
        def waiting_handler(result):
            """ waiting_content reply handler """
            self.assertEquals(2, len(result))
            # the second time we're called, the result should be reversed
            if self.second:
                node_b, node_a = result
            else:
                node_a, node_b = result

            self.assertEquals(a_path, str(node_a['path']))
            self.assertEquals(b_path, str(node_b['path']))
            self.assertEquals('share_id', str(node_a['share']))
            self.assertEquals('share_id', str(node_b['share']))
            self.assertEquals('node_id_a', str(node_a['node']))
            self.assertEquals('node_id_b', str(node_b['node']))

            if self.second:
                # OK, we're done
                d.callback(True)
            else:
                # the last shall be the first, et cetera.
                client.call_method('schedule_next', 'share_id', 'node_id_b',
                                   reply_handler=reschedule_handler,
                                   error_handler=self.error_handler)
        def reschedule_handler(result):
            """ schedule_next reply handler """
            self.assertEqual(result, None)
            self.second = True
            client.call_method('waiting_content',
                               reply_handler=waiting_handler,
                               error_handler=self.error_handler)
        client.call_method('waiting_content',
                           reply_handler=waiting_handler,
                           error_handler=self.error_handler)
        return d

    def test_contq_changed(self):
        '''
        Test the Status.ContentQueueChanged signal
        '''
        class FakeCommand(object):
            '''
            A fake command
            '''
            def __init__(self, share_id, node_id):
                self.share_id = share_id
                self.node_id = node_id
            def is_runnable(self):
                '''is runnable'''
                return True
            def run(self):
                '''just succeed'''
                return defer.succeed(None)
        # prepare the VM so it lies for us
        share_path = os.path.join(self.shares_dir, 'share')
        self.main.vm.add_share(Share(share_path, share_id='share_id'))
        a_path = os.path.join(share_path, "path_a")
        self.fs_manager.create(a_path, "share_id")
        self.fs_manager.set_node_id(a_path, "node_id")
        # OK, testing time
        d = defer.Deferred()
        def content_queue_change_helepr(result):
            '''
            signal handler, test checker
            '''
            head_path = result['head']['path']
            assert head_path == '' or head_path.endswith('/path_a')
            assert result['head']['command'] == 'FakeCommand'
            assert result.get('FakeCommand') in (None, {'size': '0',
                                                        'count': '1'})
            d.callback(None)
        match = self.bus.add_signal_receiver(content_queue_change_helepr,
                                             'ContentQueueChanged')
        self.signal_receivers.add(match)
        # inject the fake data
        self.action_q.content_queue.queue(FakeCommand("share_id", "node_id"))
        return d

    def test_current_downloads(self):
        """ Test Status.current_downloads with fake data in the AQ """
        share_path = os.path.join(self.shares_dir, 'share')
        self.main.vm.add_share(Share(share_path, share_id='share_id'))
        down_path = os.path.join(share_path, "down_path")
        self.fs_manager.create(down_path, "share_id")
        self.fs_manager.set_node_id(down_path, "node_id")
        self.action_q.downloading[('share_id', 'node_id')] = dict(
            deflated_size=10, size=100, n_bytes_read=1)
        client = DBusClient(self.bus, '/status', DBUS_IFACE_STATUS_NAME)
        d = defer.Deferred()
        def downloads_handler(result):
            """ reply handler """
            self.assertEquals(1, len(result))
            self.assertEquals(down_path, str(result[0]['path']))
            self.assertEquals('10', str(result[0]['deflated_size']))
            self.assertEquals('1', str(result[0]['n_bytes_read']))
            d.callback(True)

        client.call_method('current_downloads',
                           reply_handler=downloads_handler,
                           error_handler=self.error_handler)
        return d

    def test_current_uploads(self):
        """ Test Status.current_uploads with fake data in the AQ """
        share_path = os.path.join(self.shares_dir, 'share')
        self.main.vm.add_share(Share(share_path, share_id='share_id'))
        up_path = os.path.join(share_path, "up_path")
        self.fs_manager.create(up_path, "share_id")
        self.fs_manager.set_node_id(up_path, "node_id")
        self.action_q.uploading[('share_id', 'node_id')] = dict(
            deflated_size=100, n_bytes_written=10)
        client = DBusClient(self.bus, '/status', DBUS_IFACE_STATUS_NAME)
        d = defer.Deferred()

        def uploads_handler(result):
            """ reply handler """
            self.assertEquals(1, len(result))
            self.assertEquals(up_path, str(result[0]['path']))
            self.assertEquals('100', str(result[0]['deflated_size']))
            self.assertEquals('10', str(result[0]['n_bytes_written']))
            d.callback(True)

        client.call_method('current_uploads', reply_handler=uploads_handler,
                           error_handler=self.error_handler)
        return d

    def test_current_uploads_with_marker(self):
        """ Test Status.current_uploads with fake data in the AQ """
        self.action_q.uploading[('', MDMarker('a_marker_uuid'))] = dict(
            deflated_size=100, n_bytes_written=10)
        client = DBusClient(self.bus, '/status', DBUS_IFACE_STATUS_NAME)
        d = defer.Deferred()

        def uploads_handler(result):
            """ reply handler """
            self.assertEquals(0, len(result))
            d.callback(True)

        client.call_method('current_uploads', reply_handler=uploads_handler,
                           error_handler=self.error_handler)
        return d

    def test_two_current_downloads(self):
        """ Test Status.current_downloads with fake data in the AQ """
        share_path = os.path.join(self.shares_dir, 'share')
        self.main.vm.add_share(Share(share_path, share_id='share_id'))
        down_path = os.path.join(share_path, "down_path")
        self.fs_manager.create(down_path, "share_id")
        self.fs_manager.set_node_id(down_path, "node_id")
        self.action_q.downloading[('share_id', 'node_id')] = dict(
            deflated_size=10, size=100, n_bytes_read=8)
        share1_path = os.path.join(self.shares_dir, 'share1')
        self.main.vm.add_share(Share(share1_path, share_id='share_id_1'))
        down_path_1 = os.path.join(share1_path, "down_path_1")
        self.fs_manager.create(down_path_1, "share_id_1")
        self.fs_manager.set_node_id(down_path_1, "node_id_1")
        self.action_q.downloading[('share_id_1', 'node_id_1')] = dict(
            deflated_size=10, size=100, n_bytes_read=5)
        client = DBusClient(self.bus, '/status', DBUS_IFACE_STATUS_NAME)
        d = defer.Deferred()
        def downloads_handler(result):
            """ reply handler """
            self.assertEquals(2, len(result))
            self.assertEquals(down_path, str(result[0]['path']))
            self.assertEquals('10', str(result[0]['deflated_size']))
            self.assertEquals('8', str(result[0]['n_bytes_read']))
            self.assertEquals(down_path_1, str(result[1]['path']))
            self.assertEquals('10', str(result[1]['deflated_size']))
            self.assertEquals('5', str(result[1]['n_bytes_read']))
            d.callback(True)

        client.call_method('current_downloads',
                           reply_handler=downloads_handler,
                           error_handler=self.error_handler)
        return d

    def test_two_current_uploads(self):
        """ Test Status.current_uploads with fake data in the AQ """
        share_path = os.path.join(self.shares_dir, 'share')
        self.main.vm.add_share(Share(share_path, share_id='share_id'))
        up_path = os.path.join(share_path, "up_path")
        self.fs_manager.create(up_path, "share_id")
        self.fs_manager.set_node_id(up_path, "node_id")
        self.action_q.uploading[('share_id', 'node_id')] = dict(
            deflated_size=100, n_bytes_written=10)
        share1_path = os.path.join(self.shares_dir, 'share1')
        self.main.vm.add_share(Share(share1_path, share_id='share_id_1'))
        up_path_1 = os.path.join(share1_path, "up_path_1")
        self.fs_manager.create(up_path_1, "share_id_1")
        self.fs_manager.set_node_id(up_path_1, "node_id_1")
        self.action_q.uploading[('share_id_1', 'node_id_1')] = dict(
            deflated_size=80, n_bytes_written=20)
        client = DBusClient(self.bus, '/status', DBUS_IFACE_STATUS_NAME)
        d = defer.Deferred()

        def uploads_handler(result):
            """ reply handler """
            self.assertEquals(2, len(result))
            self.assertEquals(up_path, str(result[0]['path']))
            self.assertEquals('100', str(result[0]['deflated_size']))
            self.assertEquals('10', str(result[0]['n_bytes_written']))
            self.assertEquals(up_path_1, str(result[1]['path']))
            self.assertEquals('80', str(result[1]['deflated_size']))
            self.assertEquals('20', str(result[1]['n_bytes_written']))
            d.callback(True)

        client.call_method('current_uploads', reply_handler=uploads_handler,
                           error_handler=self.error_handler)
        return d

    def test_current_downloads_deflated_size_NA(self):
        """ Test Status.current_downloads with fake data in the AQ """
        share_path = os.path.join(self.shares_dir, 'share')
        self.main.vm.add_share(Share(share_path, share_id='share_id'))
        down_path = os.path.join(share_path, "down_path")
        self.fs_manager.create(down_path, "share_id")
        self.fs_manager.set_node_id(down_path, "node_id")
        self.action_q.downloading[('share_id', 'node_id')] = dict()
        client = DBusClient(self.bus, '/status', DBUS_IFACE_STATUS_NAME)
        d = defer.Deferred()
        def downloads_handler(result):
            """ reply handler """
            self.assertEquals(1, len(result))
            self.assertEquals(down_path, str(result[0]['path']))
            self.assertNotIn('deflated_size', result[0])
            self.assertEquals('0', str(result[0]['n_bytes_read']))
            d.callback(True)

        client.call_method('current_downloads',
                           reply_handler=downloads_handler,
                           error_handler=self.error_handler)
        return d

    def test_current_uploads_deflated_size_NA(self):
        """ Test Status.current_uploads with fake data in the AQ """
        share_path = os.path.join(self.shares_dir, 'share')
        self.main.vm.add_share(Share(share_path, share_id='share_id'))
        up_path = os.path.join(share_path, "up_path")
        self.fs_manager.create(up_path, "share_id")
        self.fs_manager.set_node_id(up_path, "node_id")
        self.action_q.uploading[('share_id', 'node_id')] = dict()
        client = DBusClient(self.bus, '/status', DBUS_IFACE_STATUS_NAME)
        d = defer.Deferred()

        def uploads_handler(result):
            """ reply handler """
            self.assertEquals(1, len(result))
            self.assertEquals(up_path, str(result[0]['path']))
            self.assertNotIn('deflated_size', result[0])
            self.assertEquals('0', str(result[0]['n_bytes_written']))
            d.callback(True)

        client.call_method('current_uploads', reply_handler=uploads_handler,
                           error_handler=self.error_handler)
        return d

    def test_nm_signals(self):
        """ Test that NM signals are received and handled properly. """
        result = None
        d = defer.Deferred()
        # helper class, pylint: disable-msg=C0111
        # class-closure, cannot use self, pylint: disable-msg=E0213
        class Listener(object):
            def handle_SYS_NET_CONNECTED(innerself):
                self.nm.emit_disconnected()
            def handle_SYS_NET_DISCONNECTED(innerself):
                self.assertTrue(result)

        listener = Listener()
        self.event_q.subscribe(listener)

        def empty_queue_callback():
            d.callback(True)

        def callback(result):
            self.event_q.unsubscribe(listener)
            self.event_q.remove_empty_event_queue_callback(empty_queue_callback)
        d.addCallback(callback)

        self.nm.emit_connected()
        self.event_q.add_empty_event_queue_callback(empty_queue_callback)
        return d

    def test_get_shares(self):
        """ Test Shares.get_shares method"""
        share_path = os.path.join(self.main.shares_dir, 'share')
        self.main.vm.add_share(Share(path=share_path, share_id='share_id',
                                     access_level='Read', accepted=False))
        client = DBusClient(self.bus, '/shares', DBUS_IFACE_SHARES_NAME)
        d = defer.Deferred()
        def shares_handler(shares):
            """ handle get_shares reply """
            self.assertEquals(1, len(shares))
            for share in shares:
                if share['id'] == '':
                    self.assertEquals('', str(share['id']))
                    self.assertEquals(self.root_dir, str(share['path']))
                    self.assertEquals('Modify', str(share['access_level']))
                    self.assertEquals('False', str(share['accepted']))
                else:
                    self.assertEquals('share_id', str(share['id']))
                    self.assertEquals(share_path, str(share['path']))
                    self.assertEquals('Read', str(share['access_level']))
                    self.assertEquals('False', str(share['accepted']))
            d.callback(True)

        client.call_method('get_shares', reply_handler=shares_handler,
                           error_handler=self.error_handler)
        return d

    def test_accept_share(self):
        """ Test the accept_share method in dbus_interface.Share """
        share_path = os.path.join(self.main.shares_dir, 'share')
        self.main.vm.add_share(Share(path=share_path, share_id='share_id',
                                     access_level='Read', accepted=False,
                                     subtree="node_id"))
        self.assertEquals(False, self.main.vm.shares['share_id'].accepted)
        client = DBusClient(self.bus, '/shares', DBUS_IFACE_SHARES_NAME)

        d = defer.Deferred()
        match = self.bus.add_signal_receiver(d.callback,
                                             signal_name='ShareAnswerResponse')
        self.signal_receivers.add(match)
        client.call_method('accept_share', 'share_id',
                           reply_handler=lambda _: None,
                           error_handler=self.error_handler)
        def check(result):
            """the async checker"""
            self.assertEquals('Yes', result['answer'])
            self.assertEquals('share_id', result['share_id'])
            self.assertEquals(True, self.main.vm.shares['share_id'].accepted)

        d.addCallback(check)
        return d

    def test_reject_share(self):
        """ Test the reject_share method in dbus_interface.Share """
        share_path = os.path.join(self.main.shares_dir, 'share')
        self.main.vm.add_share(Share(path=share_path, share_id='share_id',
                                     access_level='Read', accepted=False))
        self.assertEquals(False, self.main.vm.shares['share_id'].accepted)
        client = DBusClient(self.bus, '/shares', DBUS_IFACE_SHARES_NAME)

        d = defer.Deferred()
        match = self.bus.add_signal_receiver(d.callback,
                                             signal_name='ShareAnswerResponse')
        self.signal_receivers.add(match)
        client.call_method('reject_share', 'share_id',
                           reply_handler=lambda _: None,
                           error_handler=self.error_handler)
        def check(result):
            """the async checker"""
            self.assertEquals('No', result['answer'])
            self.assertEquals('share_id', result['share_id'])
            self.assertEquals(False, self.main.vm.shares['share_id'].accepted)

        d.addCallback(check)
        return d

    def test_get_root(self):
        """ Check SycnDaemon.get_root exposed method """
        client = DBusClient(self.bus, '/', DBUS_IFACE_SYNC_NAME)
        d = defer.Deferred()
        def reply_handler(result):
            """ handle get_rootdir reply """
            current_root = self.main.get_rootdir()
            self.assertEquals(current_root, str(result))
            d.callback(True)
        client.call_method('get_rootdir',
                           reply_handler=reply_handler,
                           error_handler=self.error_handler)
        return d

    def test_create_share(self):
        """ Test share offering """
        a_dir = os.path.join(self.root_dir, "a_dir")
        self.fs_manager.create(a_dir, "", is_dir=True)
        self.fs_manager.set_node_id(a_dir, "node_id")
        client = DBusClient(self.bus, '/shares', DBUS_IFACE_SHARES_NAME)
        d = defer.Deferred()
        # helper functions, pylint: disable-msg=C0111
        def fake_create_share(*result):
            # pylint: disable-msg=W0612
            node_id, username, name, access_level, marker = result
            self.assertEquals('node_id', node_id)
            mdobj = self.fs_manager.get_by_path(a_dir)
            self.assertEquals(self.fs_manager.get_abspath("", mdobj.path),
                              os.path.join(self.main.root_dir, a_dir))
            self.assertEquals(u'test_user', username)
            self.assertEquals(u'share_a_dir', name)
            self.assertEquals('View', access_level)

        self.action_q.create_share = fake_create_share

        def reply_handler(result):
            d.callback(result)

        client.call_method('create_share',
                           a_dir, 'test_user',
                           'share_a_dir', 'View',
                           reply_handler=reply_handler,
                           error_handler=self.error_handler)
        return d

    def test_query_by_path(self):
        """ test that query_by_path method work as expected. """
        a_dir = os.path.join(self.root_dir, "a_dir")
        self.fs_manager.create(a_dir, "", is_dir=True)
        self.fs_manager.set_node_id(a_dir, "node_id")
        client = DBusClient(self.bus, '/', DBUS_IFACE_SYNC_NAME)
        d = defer.Deferred()
        # helper functions, pylint: disable-msg=C0111
        def query(items):
            self.assertEquals(1, len(items))
            self.assertEquals(('', 'node_id', ''), items[0])
        self.action_q.query = query
        client.call_method('query_by_path', a_dir,
                           reply_handler=d.callback,
                           error_handler=self.error_handler)
        return d

    def test_get_shared(self):
        """ Test Shares.get_shared API. """
        a_dir = os.path.join(self.root_dir, "a_dir")
        self.fs_manager.create(a_dir, "", is_dir=True)
        self.fs_manager.set_node_id(a_dir, "node_id")
        # helper functions, pylint: disable-msg=C0111
        def aq_create_share(*args):
            self.main.event_q.push('AQ_CREATE_SHARE_OK', 'share_id', args[-1])
        self.main.action_q.create_share = aq_create_share
        self.main.vm.create_share(a_dir, 0, 'share_a_dir', 'View')
        client = DBusClient(self.bus, '/shares', DBUS_IFACE_SHARES_NAME)
        d = defer.Deferred()
        def reply_handler(results):
            """ handle get_shared reply """
            self.assertEquals(1, len(results))
            shared = results[0]
            self.assertEquals(a_dir, str(shared['path']))
            self.assertEquals('node_id', str(shared['subtree']))
            self.assertEquals('share_id', str(shared['id']))
            self.assertEquals('View', str(shared['access_level']))
            d.callback(True)
        client.call_method('get_shared',
                           reply_handler=reply_handler,
                           error_handler=self.error_handler)
        return d

    def test_get_shared_missing_path(self):
        """ Test Shares.get_shared, but without having the path. """
        a_dir = os.path.join(self.root_dir, "a_dir")
        self.fs_manager.create(a_dir, "", is_dir=True)
        self.fs_manager.set_node_id(a_dir, "node_id")
        # helper functions, pylint: disable-msg=C0111
        def aq_create_share(*args):
            self.main.event_q.push('AQ_CREATE_SHARE_OK', 'share_id', args[-1])
        self.main.action_q.create_share = aq_create_share
        self.main.vm.create_share(a_dir, 0, 'share_a_dir', 'View')
        client = DBusClient(self.bus, '/shares', DBUS_IFACE_SHARES_NAME)
        # remove the md of the subtree from fsm
        self.fs_manager.delete_file(a_dir)
        # set the path to None
        share = self.main.vm.shared['share_id']
        share.path = None
        self.main.vm.shared['share_id'] = share
        d = defer.Deferred()
        def reply_handler(results):
            """ handle get_shared reply """
            self.assertEquals(1, len(results))
            shared = results[0]
            self.assertEquals('', str(shared['path']))
            self.assertEquals('node_id', str(shared['subtree']))
            self.assertEquals('share_id', str(shared['id']))
            self.assertEquals('View', str(shared['access_level']))
            d.callback(True)
        client.call_method('get_shared',
                           reply_handler=reply_handler,
                           error_handler=self.error_handler)
        return d
    def test_refresh_shares(self):
        """ Just check that refresh_shares method API works. """
        client = DBusClient(self.bus, '/shares', DBUS_IFACE_SHARES_NAME)
        d = defer.Deferred()
        def reply_handler(result):
            """ handle get_rootdir reply """
            d.callback(True)
        client.call_method('refresh_shares',
                           reply_handler=reply_handler,
                           error_handler=self.error_handler)
        return d

    def test_quit(self):
        """ test the quit exposed method. """
        client = DBusClient(self.bus, '/', DBUS_IFACE_SYNC_NAME)
        d = defer.Deferred()
        # helper functions, we need to call quit but don't quit
        # pylint: disable-msg=C0111,W0601,W0602
        def fake_quit():
            pass
        self.main.quit = fake_quit
        def handler(result):
            d.callback(True)
        d.addCallback(self.assertTrue)
        client.call_method('quit',
                           reply_handler=handler,
                           error_handler=self.error_handler)
        return d


class DBusInterfaceUnicodeTests(DBusTwistedTestCase):
    """ Unicode variant of basic tests to the objects exposed with D-Bus"""

    def test_filesystem_unicode(self):
        """ Test the FileSystem Object, registering it to the session bus, and
        excercising the API.
        """
        share_path = os.path.join(self.shares_dir, 'share')
        self.main.vm.add_share(Share(share_path, share_id='share'))
        path = os.path.join(share_path, u'ñoño'.encode('utf-8'))
        self.fs_manager.create(path, "share")
        self.fs_manager.set_node_id(path, "node_id")
        d = defer.Deferred()
        def handler(metadata):
            """ reply handler """
            d.callback(metadata)

        def callback(result):
            """ callback to check the result. """
            self.assertEquals(u'\xf1o\xf1o', unicode(result['path']))
            self.assertEquals('share', result['share_id'])
            self.assertEquals('node_id', result['node_id'])

        d.addCallback(callback)
        client = DBusClient(self.bus, '/filesystem', DBUS_IFACE_FS_NAME)
        client.call_method('get_metadata', path, reply_handler=handler,
                           error_handler=self.error_handler)
        return d

    def test_query_by_path_unicode(self):
        """ test that query_by_path method work as expected. """
        a_dir = os.path.join(self.root_dir, u'ñoño'.encode('utf-8'))
        self.fs_manager.create(a_dir, "", is_dir=True)
        self.fs_manager.set_node_id(a_dir, "node_id")
        client = DBusClient(self.bus, '/', DBUS_IFACE_SYNC_NAME)
        d = defer.Deferred()
        # helper functions, pylint: disable-msg=C0111
        def query(items):
            self.assertEquals(1, len(items))
            self.assertEquals(('', 'node_id', ''), items[0])
        self.action_q.query = query
        client.call_method('query_by_path', a_dir,
                           reply_handler=d.callback,
                           error_handler=self.error_handler)
        return d

    def test_create_share_unicode(self):
        """ Test share offering """
        a_dir = os.path.join(self.root_dir, u'ñoño'.encode('utf-8'))
        self.fs_manager.create(a_dir, "", is_dir=True)
        self.fs_manager.set_node_id(a_dir, "node_id")
        client = DBusClient(self.bus, '/shares', DBUS_IFACE_SHARES_NAME)
        d = defer.Deferred()
        # helper functions, pylint: disable-msg=C0111
        def fake_create_share(*result):
            # pylint: disable-msg=W0612
            node_id, username, name, access_level, marker = result
            self.assertEquals('node_id', node_id)
            mdobj = self.fs_manager.get_by_path(a_dir)
            self.assertEquals(self.fs_manager.get_abspath("", mdobj.path),
                              os.path.join(self.root_dir, a_dir))
            self.assertEquals(u'test_user', username)
            self.assertEquals(u'share_ñoño', name)
            self.assertEquals('View', access_level)

        self.action_q.create_share = fake_create_share

        def reply_handler(result):
            d.callback(result)

        client.call_method('create_share',
                           a_dir, u'test_user',
                           u'share_ñoño', 'View',
                           reply_handler=reply_handler,
                           error_handler=self.error_handler)
        return d

    def test_get_shared_unicode(self):
        """ test that list_shared method behaves with unicode data """
        a_dir = os.path.join(self.root_dir, u'ñoño'.encode('utf-8'))
        self.fs_manager.create(a_dir, "", is_dir=True)
        self.fs_manager.set_node_id(a_dir, "node_id")
        share = Share(a_dir, share_id='shared_id', name=u'ñoño_shared',
                      access_level='View', other_username=u'test_username',
                      subtree='node_id')
        self.main.vm.add_shared(share)
        client = DBusClient(self.bus, '/shares', DBUS_IFACE_SHARES_NAME)
        d = defer.Deferred()

        def check(results):
            """ check the returned values. """
            self.assertEquals(1, len(results))
            shared = results[0]
            self.assertEquals(a_dir, shared['path'].encode('utf-8'))
            self.assertEquals('node_id', str(shared['subtree']))
            self.assertEquals('shared_id', str(shared['id']))
            self.assertEquals('View', str(shared['access_level']))

        d.addCallback(check)
        client.call_method('get_shared',
                           reply_handler=d.callback, error_handler=d.errback)
        return d


class DBusSignalTest(DBusTwistedTestCase):
    """ Test the DBus signals. """

    def test_event_signal(self):
        """ Test the Event signal. """
        client = DBusClient(self.bus, '/events', DBUS_IFACE_EVENTS_NAME)
        d = defer.Deferred()
        def handler(event_dict):
            """ signal handler """
            self.assertEquals({'path':'foo', 'name':'FS_FILE_CREATE'},
                              event_dict)
            d.callback(True)

        match = self.bus.add_signal_receiver(handler, signal_name='Event')
        self.signal_receivers.add(match)
        client.send_signal('Event', {'name':'FS_FILE_CREATE', 'path':'foo'})

        return d

    def test_download_started(self):
        """ Test the DBus signals in Status """
        a_dir = os.path.join(self.root_dir, u'ñoño'.encode('utf-8'))
        self.fs_manager.create(a_dir, "", is_dir=False)
        self.fs_manager.set_node_id(a_dir, "node_id")

        d = defer.Deferred()
        def download_handler(path):
            """ handler for DownloadStarted signal. """
            self.assertEquals(a_dir, path.encode('utf-8'))
            d.callback(True)

        match = self.bus.add_signal_receiver(download_handler,
                                     signal_name='DownloadStarted')
        self.signal_receivers.add(match)
        self.main.event_q.push('AQ_DOWNLOAD_STARTED', '', 'node_id', '')
        return d

    def test_download_finished(self):
        """ Test the DBus signals in Status """
        a_dir = os.path.join(self.root_dir, u'ñoño'.encode('utf-8'))
        self.fs_manager.create(a_dir, "", is_dir=False)
        self.fs_manager.set_node_id(a_dir, "node_id")

        d = defer.Deferred()
        def download_handler(path, info):
            """ handler for DownloadFinished signal. """
            self.assertEquals(a_dir, path.encode('utf-8'))
            d.callback(True)

        match = self.bus.add_signal_receiver(download_handler,
                                     signal_name='DownloadFinished')
        self.signal_receivers.add(match)
        self.main.event_q.push('AQ_DOWNLOAD_FINISHED', '', 'node_id', '')
        return d

    def test_download_error(self):
        """ Test the DBus signals in Status """
        a_dir = os.path.join(self.root_dir, u'ñoño'.encode('utf-8'))
        self.fs_manager.create(a_dir, "", is_dir=False)
        self.fs_manager.set_node_id(a_dir, "node_id")

        d = defer.Deferred()
        def download_handler(path, info):
            """ handler for DownloadFinished signal. """
            self.assertEquals(a_dir, path.encode('utf-8'))
            self.assertEquals('AN_ERROR', info['error'])
            d.callback(True)

        match = self.bus.add_signal_receiver(download_handler,
                                     signal_name='DownloadFinished')
        self.signal_receivers.add(match)
        self.main.event_q.push('AQ_DOWNLOAD_ERROR',
                               '', 'node_id', '', error='AN_ERROR')
        return d

    def test_download_cancelled(self):
        """ Test the DBus signals in Status """
        a_dir = os.path.join(self.root_dir, u'ñoño'.encode('utf-8'))
        self.fs_manager.create(a_dir, "", is_dir=False)
        self.fs_manager.set_node_id(a_dir, "node_id")

        d = defer.Deferred()
        def download_handler(path, info):
            """ handler for DownloadFinished signal. """
            self.assertEquals(a_dir, path.encode('utf-8'))
            self.assertEquals('CANCELLED', info['error'])
            d.callback(True)

        match = self.bus.add_signal_receiver(download_handler,
                                     signal_name='DownloadFinished')
        self.signal_receivers.add(match)
        self.main.event_q.push('AQ_DOWNLOAD_CANCELLED', '', 'node_id', 'hash')
        return d

    def test_upload_started(self):
        """ Test the DBus signals in Status """
        a_dir = os.path.join(self.root_dir, u'ñoño'.encode('utf-8'))
        self.fs_manager.create(a_dir, "", is_dir=False)
        self.fs_manager.set_node_id(a_dir, "node_id")

        d = defer.Deferred()
        def upload_handler(path):
            """ handler for UploadStarted signal. """
            self.assertEquals(a_dir, path.encode('utf-8'))
            d.callback(True)

        match = self.bus.add_signal_receiver(upload_handler,
                                     signal_name='UploadStarted')
        self.signal_receivers.add(match)
        self.main.event_q.push('AQ_UPLOAD_STARTED', '', 'node_id', '')
        return d

    def test_upload_finished(self):
        """ Test the DBus signals in Status """
        a_dir = os.path.join(self.root_dir, u'ñoño'.encode('utf-8'))
        self.fs_manager.create(a_dir, "", is_dir=False)
        self.fs_manager.set_node_id(a_dir, "node_id")

        d = defer.Deferred()
        def upload_handler(path, info):
            """ handler for UploadFinished signal. """
            self.assertEquals(a_dir, path.encode('utf-8'))
            d.callback(True)

        match = self.bus.add_signal_receiver(upload_handler,
                                     signal_name='UploadFinished')
        self.signal_receivers.add(match)
        self.main.event_q.push('AQ_UPLOAD_FINISHED', '', 'node_id', '')
        return d

    def test_upload_error(self):
        """ Test the DBus signals in Status """
        a_dir = os.path.join(self.root_dir, u'ñoño'.encode('utf-8'))
        self.fs_manager.create(a_dir, "", is_dir=False)
        self.fs_manager.set_node_id(a_dir, "node_id")

        d = defer.Deferred()
        def upload_handler(path, info):
            """ handler for UploadFinished signal. """
            self.assertEquals(a_dir, path.encode('utf-8'))
            self.assertEquals('AN_ERROR', info['error'])
            d.callback(True)

        match = self.bus.add_signal_receiver(upload_handler,
                                     signal_name='UploadFinished')
        self.signal_receivers.add(match)
        self.main.event_q.push('AQ_UPLOAD_ERROR',
                               '', 'node_id', 'AN_ERROR', 'hash')
        return d

    def test_status_changed(self):
        """ Test the DBus signals in Status """
        client = DBusClient(self.bus, '/status', DBUS_IFACE_STATUS_NAME)
        def start(result):
            """ start the test """
            match = self.bus.add_signal_receiver(status_handler,
                                     signal_name='StatusChanged')
            self.signal_receivers.add(match)
            bool_str = self.dbus_iface.status.bool_str
            state = states.IDLE
            state_dict = {'name':state.name,
                          'description':state.description,
                          'is_error':bool_str(state.is_error),
                          'is_connected':bool_str(state.is_connected),
                          'is_online':bool_str(state.is_online)}
            client.send_signal('StatusChanged', state_dict)
        ready = self.main.wait_for_nirvana()
        ready.addCallback(start)
        d = defer.Deferred()
        def status_handler(result):
            """ handler for StatusChanged signal. """
            self.log.debug('status_handler, received: %r' % result)
            state = states.IDLE
            self.assertEquals(state.name, result['name'])
            self.assertEquals(state.description, result['description'])
            self.assertEquals(state.is_error, bool(result['is_error']))
            self.assertEquals(state.is_connected, bool(result['is_connected']))
            self.assertEquals(state.is_online, bool(result['is_online']))
            d.callback(True)

        return d

    def test_all_events_signal(self):
        """ Tests that we are listening all events """
        # configure DBusInterface to send signal for the events
        self.dbus_iface.send_events = True
        # create the required items in the fs manager
        share_path = os.path.join(self.shares_dir, 'share')
        self.main.vm.add_share(Share(share_path, share_id='share_id'))
        a_path = os.path.join(share_path, "a_path")
        self.fs_manager.create(a_path, "share_id")
        self.fs_manager.set_node_id(a_path, "node_id")
        handled_events = {}
        # filter only event we handle
        for (name, kind, _, _) in \
                                  inspect.classify_class_attrs(EventListener):
            if kind == 'method' and name.startswith('handle_') \
               and not name == 'handle_default':
                key = name.split('handle_')[1]
                handled_events[key] = event_queue.EVENTS[key]

        event_names = sorted(handled_events.keys())
        events_copy = handled_events.copy()
        d = defer.Deferred()
        def event_handler(event_dict):
            """ event handler that pop the received event from a list """
            event_args = events_copy.pop(str(event_dict.pop('event_name')))
            self.assertEquals(len(event_args), len(event_dict))
            if len(events_copy) == 0:
                cleanup(None)
                d.callback(True)

        def callback(result):
            """ default callback """
            return result

        def errback(error):
            """ error callback """
            missed_events = sorted(events_copy.keys())
            if missed_events:
                self.fail('Missed: ' + str(missed_events))
            else:
                return error

        d.addCallbacks(callback, errback)
        def cleanup(result):
            """ remove the signal receiver """
            self.event_q.subscribe(self.main.vm)
        d.addCallback(cleanup)
        # unsubscribe the vm subscribed in FakeMain as we are going to push
        # fake events
        self.event_q.unsubscribe(self.main.vm)
        match = self.bus.add_signal_receiver(event_handler, signal_name='Event')
        self.signal_receivers.add(match)
        for event_name in event_names:
            args = event_queue.EVENTS[event_name]
            self.event_q.push(event_name, *args)
        return d

    def test_share_changed(self):
        """ Test the ShareChanged signal. """
        share_path = os.path.join(self.main.shares_dir, 'share')
        self.main.vm.add_share(Share(path=share_path, share_id='a_share_id',
                                     access_level='Read', accepted=False))
        share_holder = NotifyShareHolder.from_params('a_share_id', 'subtree',
                                                     u'fake_share',
                                                     u'test_username',
                                                     u'visible_name', 'Write')

        d = defer.Deferred()
        def share_handler(share):
            """ handler for ShareChanged signal. """
            self.assertEquals('a_share_id', str(share['id']))
            self.assertEquals(share_path, str(share['path']))
            self.assertEquals('Write', str(share['access_level']))
            self.assertEquals('False', str(share['accepted']))
            d.callback(True)

        match = self.bus.add_signal_receiver(share_handler,
                                     signal_name='ShareChanged')
        self.signal_receivers.add(match)
        self.main.event_q.push('SV_SHARE_CHANGED', 'changed', share_holder)
        return d

    def test_share_deleted(self):
        """ Test the ShareDeleted signal. """
        share_path = os.path.join(self.main.shares_dir, 'share')
        self.main.vm.add_share(Share(path=share_path, share_id='a_share_id',
                                     access_level='Read', accepted=False))
        share_holder = NotifyShareHolder.from_params('a_share_id', 'subtree',
                                                     u'fake_share',
                                                     u'test_username',
                                                     u'visible_name', 'Read')

        d = defer.Deferred()
        def share_handler(share_dict):
            """ handler for ShareDeletedsignal. """
            expected_dict = dict(share_id='a_share_id',
                                 subtree='subtree',
                                 share_name=u'fake_share',
                                 from_username=u'test_username',
                                 from_visible_name=u'visible_name',
                                 access_level='Read')
            for k, v in share_dict.items():
                self.assertEquals(expected_dict[str(k)], str(v))
            d.callback(True)

        match = self.bus.add_signal_receiver(share_handler,
                                     signal_name='ShareDeleted')
        self.signal_receivers.add(match)

        self.main.event_q.push('SV_SHARE_CHANGED', 'deleted', share_holder)
        return d

    def test_share_created(self):
        """ Test the ShareCreated signal. """
        a_dir = os.path.join(self.root_dir, "a_dir")
        self.fs_manager.create(a_dir, "", is_dir=True)
        self.fs_manager.set_node_id(a_dir, "node_id")
        mdobj = self.fs_manager.get_by_node_id("", "node_id")
        mdid = mdobj.mdid
        marker = MDMarker(mdid)
        d = defer.Deferred()
        def share_handler(result):
            """ handler for ShareCreated signal. """
            self.assertEquals('share_id', str(result['share_id']))
            d.callback(True)

        match = self.bus.add_signal_receiver(share_handler,
                                     signal_name='ShareCreated')
        self.signal_receivers.add(match)
        self.main.event_q.push('AQ_CREATE_SHARE_OK', 'share_id', marker)
        return d

    def test_share_created_error(self):
        """ Test the ShareCreated signal. """
        a_dir = os.path.join(self.root_dir, "a_dir")
        self.fs_manager.create(a_dir, "", is_dir=True)
        self.fs_manager.set_node_id(a_dir, "node_id")
        mdobj = self.fs_manager.get_by_node_id("", "node_id")
        mdid = mdobj.mdid
        marker = MDMarker(mdid)
        error_msg = 'a error message'
        d = defer.Deferred()
        def share_handler(info, error):
            """ handler for ShareCreated signal. """
            self.assertEquals(str(marker), str(info['marker']))
            self.assertEquals(error, error_msg)
            d.callback(True)

        match = self.bus.add_signal_receiver(share_handler,
                                     signal_name='ShareCreateError')
        self.signal_receivers.add(match)
        self.main.event_q.push('AQ_CREATE_SHARE_ERROR', marker, error_msg)
        return d

    def test_signal_error(self):
        """ test sending SignalError. """
        d = defer.Deferred()
        def error_handler(type, args):
            """ Error signal handler. """
            self.assertEquals('node_id', args['node_id'].decode('utf-8'))
            self.assertEquals('', args['share_id'].decode('utf-8'))
            d.callback(True)

        error_match = self.bus.add_signal_receiver(error_handler,
                                     signal_name='SignalError',
                                     dbus_interface=DBUS_IFACE_SYNC_NAME)
        self.signal_receivers.add(error_match)
        # force a invalid AQ_UPLOAD_FINISHED
        self.main.event_q.push('AQ_UPLOAD_FINISHED', '', 'node_id', 'hash')
        return d

class TestDBusRestart(DBusTwistedTestCase):
    """
    Test main's restart method (and its interaction with dbus).
    """
    def test_restart(self):
        """
        start things up, then fire a restart, check it tries to
        restart.
        """
        d = defer.Deferred()
        def _handler(*a):
            """async helper"""
            d.callback(True)
        # shutdown will fail when trying to restart because of our
        # half-backed dbus. That's OK, we don't actually want it
        # restarted :)
        self.dbus_iface._restart_error_handler = d.callback
        try:
            self.main.restart()
        except SystemExit:
            pass
        return d


class ConfigTests(DBusTwistedTestCase):
    """ Basic tests to the Config object exposed via D-Bus"""

    def get_client(self):
        """returns a Config DBusClient"""
        return DBusClient(self.bus, '/config', DBUS_IFACE_CONFIG_NAME)

    def test_get_throttling_limits(self):
        """test get_throttling_limits exposed method"""
        client = self.get_client()
        d = defer.Deferred()
        aq = self.main.action_q
        def reply_handler(result):
            """ handle the reply """
            self.assertEquals(aq.readLimit, result['download'])
            self.assertEquals(aq.writeLimit, result['upload'])
            self.assertEquals(100, result['download'])
            self.assertEquals(200, result['upload'])
            d.callback(True)
        def reply_handler_None(result):
            """ handle the reply """
            self.assertEquals(-1, result['download'])
            self.assertEquals(-1, result['upload'])
            aq.readLimit = 100
            aq.writeLimit = 200
            client.call_method('get_throttling_limits',
                               reply_handler=reply_handler,
                               error_handler=self.error_handler)
        client.call_method('get_throttling_limits',
                           reply_handler=reply_handler_None,
                           error_handler=self.error_handler)
        return d


    def test_set_throttling_limits(self):
        """test set_throttling_limits exposed method"""
        client = self.get_client()
        d = defer.Deferred()
        def reply_handler(_):
            """ handle the reply """
            aq = self.main.action_q
            self.assertEquals(aq.readLimit, 100)
            self.assertEquals(aq.writeLimit, 500)
            d.callback(True)
        client.call_method('set_throttling_limits', 100, 500,
                           reply_handler=reply_handler,
                           error_handler=self.error_handler)
        return d

    def test_enable_bandwidth_throttling(self):
        """test enable_bandwidth_throttling exposed method"""
        client = self.get_client()
        d = defer.Deferred()
        aq = self.main.action_q
        aq.throttling = False
        def reply_handler(_):
            """ handle the reply """
            self.assertTrue(aq.throttling)
            d.callback(True)
        client.call_method('enable_bandwidth_throttling',
                           reply_handler=reply_handler,
                           error_handler=self.error_handler)
        return d

    def test_disable_bandwidth_throttling(self):
        """test disable_bandwidth_throttling exposed method"""
        client = self.get_client()
        d = defer.Deferred()
        aq = self.main.action_q
        aq.throttling = True
        def reply_handler(_):
            """ handle the reply """
            self.assertFalse(aq.throttling)
            d.callback(True)
        client.call_method('disable_bandwidth_throttling',
                           reply_handler=reply_handler,
                           error_handler=self.error_handler)
        return d

    def test_bandwidth_throttling_enabled(self):
        """test bandwidth_throttling_enabled exposed method"""
        client = self.get_client()
        d = defer.Deferred()
        def reply_handler_enabled(result):
            """ handle the reply """
            self.assertEquals(1, result)
            d.callback(True)

        def reply_handler_disabled(result):
            """ handle the reply """
            self.assertEquals(0, result)
            self.main.action_q.throttling = True
            client.call_method('bandwidth_throttling_enabled',
                               reply_handler=reply_handler_enabled,
                               error_handler=self.error_handler)
        client.call_method('bandwidth_throttling_enabled',
                           reply_handler=reply_handler_disabled,
                           error_handler=self.error_handler)
        return d


class TestDBusOAuth(DBusTwistedTestCase):
    """Tests the interaction between dbus_interface and oauthdesktop"""

    def setUp(self):
        DBusTwistedTestCase.setUp(self)
        self.oauth = FakeLogin(self.bus)
        self._old_path = dbus_interface.DBUS_PATH_AUTH
        dbus_interface.DBUS_PATH_AUTH = '/oauthdesktop'

    def tearDown(self):
        # collect all signal receivers registered during the test
        signal_receivers = set()
        with self.bus._signals_lock:
            for group in self.bus._signal_recipients_by_object_path.values():
                for matches in group.values():
                    for match in matches.values():
                        signal_receivers.update(match)
        d = self.cleanup_signal_receivers(signal_receivers)
        def shutdown(r):
            self.oauth.shutdown()
            dbus_interface.DBUS_PATH_AUTH = self._old_path
        d.addCallback(shutdown)
        d.addCallback(lambda _: DBusTwistedTestCase.tearDown(self))
        return d

    def test_new_credentials_signal(self):
        """NewCredentials signal test"""
        self.oauth.processor.next_login_with(self.oauth.processor.got_token,
                                             ('my_token',))
        d = self.dbus_iface._request_token()
        def check(info):
            realm, key = info
            expected = (u'http://test.ubuntuone.com', u'ubuntuone')
            self.assertEquals((realm, key), expected)
        d.addCallbacks(check, self.fail)
        return d

    def test_no_credentials_signal(self):
        """NoCredentials signal test"""
        self.oauth.processor.next_login_with(self.oauth.processor.got_no_token)
        d = self.dbus_iface._request_token()
        def check(f):
            self.assertEquals(f.getErrorMessage(), 'NoCredentials')
        d.addCallbacks(self.fail, check)
        return d

    def test_auth_denied_signal(self):
        """AuthorizationDenied signal test"""
        self.oauth.processor.next_login_with(self.oauth.processor.got_denial)
        d = self.dbus_iface._request_token()
        def check(f):
            self.assertEquals(f.getErrorMessage(), 'AuthorizationDenied')
        d.addCallbacks(self.fail, check)
        return d

    def test_oauth_error_signal(self):
        """OAuthError signal test"""
        self.oauth.processor.next_login_with(self.oauth.processor.got_error,
                                            ('error!',))
        d = self.dbus_iface._request_token()
        def check(f):
            self.assertEquals(f.getErrorMessage(), 'OAuthError: error!')
        d.addCallbacks(self.fail, check)
        return d

    def test_connect_without_token(self):
        """Test connecting without having a token"""
        # replace the get_access_token method in Main
        def get_access_token():
            raise main.NoAccessToken('no token')
        self.main.get_access_token = get_access_token
        self.dbus_iface.disconnect()
        self.oauth.processor.next_login_with(self.oauth.processor.got_token,
                                             ('the token',))
        d = self.dbus_iface.connect()
        def check(result):
            self.assertEquals(result, None)
        d.addCallbacks(check, self.fail)
        return d

    def test_connect_with_token(self):
        """Test connecting with a token"""
        self.dbus_iface.disconnect()
        self.oauth.processor.next_login_with(self.oauth.processor.got_token,
                                             ('the token',))
        d = self.dbus_iface.connect()
        # check that the deferred was fired
        self.assertEquals(d.called, True)
        self.assertEquals(d.result, None)
        d.addCallbacks(lambda r: self.assertEquals(r, None), self.fail)
        return d

    def test_error_on_login(self):
        """Test connecting without having a token and getting an error in the
        call to oauthdesktop login().
        """
        # replace the get_access_token method in Main
        def get_access_token():
            raise main.NoAccessToken('no token')
        self.main.get_access_token = get_access_token
        self.dbus_iface.disconnect()
        def broken_login(*args):
            raise ValueError('oops, login is broken!')
        self.oauth.processor.next_login_with(broken_login)
        d = self.dbus_iface.connect()
        def check(result):
            self.assertIn('ValueError: oops, login is broken!', result.getErrorMessage())
        d.addCallbacks(self.fail, check)
        return d


def test_suite():
    # pylint: disable-msg=C0111
    return unittest.TestLoader().loadTestsFromName(__name__)
