#!/usr/bin/python
# -*- coding: utf-8 -*-
#
#       xorg_config.py -- example of how to customise the xorg.conf
#                         with XKit
#       
#       Copyright 2009 Alberto Milone <alberto.milone@canonical.com>
#       
#       This program is free software; you can redistribute it and/or modify
#       it under the terms of the GNU General Public License as published by
#       the Free Software Foundation; either version 2 of the License, or
#       (at your option) any later version.
#       
#       This program is distributed in the hope that it will be useful,
#       but WITHOUT ANY WARRANTY; without even the implied warranty of
#       MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#       GNU General Public License for more details.
#       
#       You should have received a copy of the GNU General Public License
#       along with this program; if not, write to the Free Software
#       Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
#       MA 02110-1301, USA.

import XKit.xutils, XKit.xorgparser
import sys
import getopt


class XorgConfig:
    source = '/etc/X11/xorg.conf'
    destination = '/etc/X11/xorg.conf'
    
    def __init__(self, custom_source=None, custom_destination=None):
        if custom_source:
            self.source = custom_source
        
        if custom_destination:
            self.destination = custom_destination
        
        try:
            self.xorg_conf = XKit.xutils.XUtils(self.source)
        except (XKit.xorgparser.ParseException, IOError):
            # if the xorg.conf doesn't exist or doesn't validate
            # start with an empty one
            self.xorg_conf = XKit.xutils.XUtils()
        
    def customiseConfig(self):
        # Does it have a ServerFlags section?  If not, create it
        if len(self.xorg_conf.globaldict['ServerFlags']) == 0:
            # Create one
            self.xorg_conf.makeSection('ServerFlags')
            
        # Set options for the ServerLayout section
        for serverflags in self.xorg_conf.globaldict['ServerFlags']:
            self.xorg_conf.addOption('ServerFlags', 'DontVTSwitch', 'True', optiontype='Option', position=serverflags)
            self.xorg_conf.addOption('ServerFlags', 'DontZap', 'True', optiontype='Option', position=serverflags)
    
        # Does it have one or more ServerLayout sections?
        if len(self.xorg_conf.globaldict['ServerLayout']) == 0:
            # Make a new ServerLayout section
            self.xorg_conf.makeSection('ServerLayout', identifier='Default Layout')

        # If no Device section is available
        if len(self.xorg_conf.globaldict['Device']) == 0:
            self.xorg_conf.makeSection('Device', identifier='Configured Video Device')
        
        # Make sure that Device sections have
        #   Option "UseFBDev" "true"
        #   Driver "intel"
        for device in self.xorg_conf.globaldict['Device']:
            self.xorg_conf.addOption('Device', 'UseFBDev', 'True', optiontype='Option', position=device)
            self.xorg_conf.setDriver('Device', 'intel', position=device)
        
        # If no Screen section is available
        if len(self.xorg_conf.globaldict['Screen']) == 0:
            self.xorg_conf.makeSection('Screen', identifier='Default Screen')
        
        # Make sure that references to the Screen sections
        # are available in the ServerLayout
        for screen in self.xorg_conf.globaldict['Screen']:
            # Add a reference to the Screen section in the ServerLayout section
            screen_name = self.xorg_conf.getIdentifier('Screen', screen)
            self.xorg_conf.addReference('ServerLayout', 'Screen', screen_name)
 
        # Does it have a DRI section?
        if len(self.xorg_conf.globaldict['DRI']) == 0:
            # Create one
            self.xorg_conf.makeSection('DRI')
        
        # Add Mode 0666 to the "DRI" section
        # there's no need to specify which DRI section
        # we're adding the option to since they have no
        # identifier
        # prefix has to be set to '' otherwise the value will be
        # put in quotation marks.
        self.xorg_conf.addOption('DRI', 'Mode', '0666', prefix='')


        # Does it have an InputDevice section whose identifier
        # is "Synaptics Touchpad" or whose driver is "synaptics"?
        input_devices = self.xorg_conf.globaldict['InputDevice']
        
        has_synaptics = False
        has_keyboard = False
        
        for section in input_devices:
            try:
                driver = self.xorg_conf.getDriver('InputDevice', section)
            except (XKit.xorgparser.OptionException, XKit.xorgparser.SectionException):
                # No driver is set
                pass
            else:
                if driver.lower() == 'synaptics':
                    # Found an InputDevice section about
                    # synaptics.
                    #
                    # Make sure that all the relevant options
                    # are applied to this section.
                    self.xorg_conf.addOption('InputDevice', 'SendCoreEvents', 'True', optiontype='Option', position=section)
                    self.xorg_conf.addOption('InputDevice', 'Device', '/dev/psaux', optiontype='Option', position=section)
                    self.xorg_conf.addOption('InputDevice', 'Protocol', 'auto-dev', optiontype='Option', position=section)
                    self.xorg_conf.addOption('InputDevice', 'SHMConfig', 'on', optiontype='Option', position=section)
                    
                    has_synaptics = True
                    
                elif driver.lower() == 'kbd':
                    # Found an InputDevice section about
                    # the keyboard.
                    # Make sure that all the relevant options
                    # are applied to this section.
                    self.xorg_conf.addOption('InputDevice', 'XkbRules', 'xorg', optiontype='Option', position=section)
                    self.xorg_conf.addOption('InputDevice', 'XkbModel', 'pc105', optiontype='Option', position=section)
                    self.xorg_conf.addOption('InputDevice', 'XkbLayout', 'us', optiontype='Option', position=section)
                    
                    has_keyboard = True
            
        if not has_synaptics:    
            # Make a new InputDevice section for the touchpad
            device_id = 'Synaptics Touchpad'
            input_device = self.xorg_conf.makeSection('InputDevice', identifier=device_id)
            
            # Add all the relevant options and the driver here
            self.xorg_conf.setDriver('InputDevice', 'synaptics', input_device)
            self.xorg_conf.addOption('InputDevice', 'SendCoreEvents', 'True', optiontype='Option', position=input_device)
            self.xorg_conf.addOption('InputDevice', 'Device', '/dev/psaux', optiontype='Option', position=input_device)
            self.xorg_conf.addOption('InputDevice', 'Protocol', 'auto-dev', optiontype='Option', position=input_device)
            self.xorg_conf.addOption('InputDevice', 'SHMConfig', 'on', optiontype='Option', position=input_device)
        
        if not has_keyboard:
            # Make a new InputDevice section for the keyboard
            device_id = 'Generic Keyboard'
            input_device = self.xorg_conf.makeSection('InputDevice', identifier=device_id)
            
            # Add all the relevant options and the driver here
            self.xorg_conf.setDriver('InputDevice', 'kbd', input_device)
            self.xorg_conf.addOption('InputDevice', 'XkbRules', 'xorg', optiontype='Option', position=input_device)
            self.xorg_conf.addOption('InputDevice', 'XkbModel', 'pc105', optiontype='Option', position=input_device)
            self.xorg_conf.addOption('InputDevice', 'XkbLayout', 'us', optiontype='Option', position=input_device)
        
        
        # Make sure that references to the InputDevice sections
        # are available in the ServerLayout
        #
        # NOTE: no duplicates will be created if references are
        #       already there
        for input_dev in self.xorg_conf.globaldict['InputDevice']:
            input_dev_name = self.xorg_conf.getIdentifier('InputDevice', input_dev)
            self.xorg_conf.addReference('ServerLayout', 'InputDevice', input_dev_name, 0)
        
        
        # Write the changes to the destination file
        self.xorg_conf.writeFile(self.destination)



def usage():
    instructionsList = ['The only accepted parameters are:'
    '\n  -i', '\tUse a custom xorg.conf to read the configuration.',
    
    '\n  -o', '\tUse a custom xorg.conf to write the configuration..',
    
    '\n  -h', '\tShow the help page.',
    ]

    print ''.join(instructionsList)

def main():
    err = 'Error: parameters not recognised'
    
    try:
        opts, args = getopt.getopt(sys.argv[1:], 'i:o:h', [])
    except getopt.GetoptError, err:
        # print help information and exit:
        sys.stderr.write(str(err)+"\n") # will print something like 'option -a not recognized'
        usage()
        sys.exit(2)
    
    
    
    # If called with no args, show the help page
#    if not opts:
#        usage()
#        sys.exit()
    
    
    source = None
    destination = None
    
    
    for o, a in opts:
        if o in ('-i'):
            source = a
        elif o in ('-o'):
            destination = a
        elif o in ('-h'):
            usage()
            sys.exit()
        else:
            assert False, 'unhandled option'
    
    xconfig = XorgConfig(custom_source=source, custom_destination=destination)
    xconfig.customiseConfig()

if __name__ == '__main__':
    main()
