/*
 * Copyright (C) 2010 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Mikkel Kamstrup Erlandsen <mikkel.kamstrup@canonical.com>
 *
 */
using Zeitgeist;
using Zeitgeist.Timestamp;
using Config;
using Gee;

namespace Unity.FilesLens {
  
  const string ICON_PATH = Config.DATADIR + "/icons/unity-icon-theme/places/svg/";
  
  public class Daemon : GLib.Object
  {
    private Zeitgeist.Log log;
    private Zeitgeist.Index index;
    private Zeitgeist.Monitor monitor;

    private Bookmarks bookmarks;
    private UrlChecker urls;

    private Unity.Lens lens;
    private Unity.Scope scope;

    /* For each section we have a set of Zeitgeist.Event templates that
     * we use to query Zeitgeist */
    private HashTable<string, Event> type_templates;

    construct
    {
      prepare_type_templates();

      scope = new Unity.Scope ("/com/canonical/unity/scope/files");
      scope.search_in_global = true;
      scope.activate_uri.connect (activate);

      lens = new Unity.Lens("/com/canonical/unity/lens/files", "files");
      lens.search_in_global = true;
      lens.search_hint = _("Search Files & Folders");
      lens.visible = true;
      populate_categories ();
      populate_filters();
      lens.add_local_scope (scope);
      
      /* Bring up Zeitgeist interfaces */
      log = new Zeitgeist.Log();
      index = new Zeitgeist.Index();
      
      /* Listen for all file:// related events from Zeitgeist */
      var templates = new PtrArray();
      var event = new Zeitgeist.Event ();
      var subject = new Zeitgeist.Subject ();
      subject.set_uri ("file://*");
      event.add_subject (subject);
      templates.add (event.ref ());
      monitor = new Zeitgeist.Monitor (new Zeitgeist.TimeRange.from_now (),
                                       (owned) templates);
      monitor.events_inserted.connect (on_zeitgeist_changed);
      monitor.events_deleted.connect (on_zeitgeist_changed);
      log.install_monitor (monitor);
      
      bookmarks = new Bookmarks ();
      urls = new UrlChecker ();

      /* Listen for filter changes */
      scope.filters_changed.connect (() => {
        scope.queue_search_changed (SearchType.DEFAULT);
      });

      scope.generate_search_key.connect ((lens_search) => {
        return lens_search.search_string.strip ();
      });

      /* Listen for changes to the lens entry search */
      scope.search_changed.connect ((lens_search, search_type, cancellable) =>
      {
        dispatch_search.begin (lens_search, search_type, cancellable);
      });

      lens.export ();
    }

    private async void dispatch_search (LensSearch lens_search,
                                        SearchType search_type,
                                        Cancellable cancellable)
    {
      if (search_type == SearchType.GLOBAL)
      {
        yield update_global_search_async (lens_search, cancellable);
      }
      else
      {
        yield update_search_async (lens_search, cancellable);
      }

      // make sure we don't forget to emit finished (if we didn't get cancelled)
      if (!cancellable.is_cancelled ())
      {
        if (lens_search.results_model.get_n_rows () == 0)
        {
          lens_search.set_reply_hint ("no-results-hint",
            _("Sorry, there are no files or folders that match your search."));
        }

        lens_search.finished ();
      }
    }

    private void populate_filters ()
    {
      var filters = new GLib.List<Unity.Filter> ();

      /* Last modified */
      {
        var filter = new RadioOptionFilter ("modified", _("Last modified"));

        filter.add_option ("last-7-days", _("Last 7 days"));
        filter.add_option ("last-30-days", _("Last 30 days"));
        filter.add_option ("last-year", _("Last year"));

        filters.append (filter);
      }

      /* Type filter */
      {
        var filter = new CheckOptionFilter ("type", _("Type"));
        filter.sort_type = OptionsFilter.SortType.DISPLAY_NAME;

        filter.add_option ("documents", _("Documents"));
        filter.add_option ("folders", _("Folders"));
        filter.add_option ("images", _("Images"));
        filter.add_option ("audio", _("Audio"));
        filter.add_option ("videos", _("Videos"));
        filter.add_option ("presentations", _("Presentations"));
        filter.add_option ("other", _("Other"));

        filters.append (filter);
      }

      /* Size filter */
      {
        var filter = new MultiRangeFilter ("size", _("Size"));

        filter.add_option ("1KB", _("1KB"));
        filter.add_option ("100KB", _("100KB"));
        filter.add_option ("1MB", _("1MB"));
        filter.add_option ("10MB", _("10MB"));
        filter.add_option ("100MB", _("100MB"));
        filter.add_option ("1GB", _("1GB"));
        filter.add_option (">1GB", _(">1GB"));

        filters.append (filter);
      }

      lens.filters = filters;
    }

    private void populate_categories ()
    {
      var categories = new GLib.List<Unity.Category> ();
      var icon_dir = File.new_for_path (ICON_PATH);

      var cat = new Unity.Category (_("Recent"),
                                    new FileIcon (icon_dir.get_child ("group-recent.svg")));
      categories.append (cat);
      
      cat = new Unity.Category (_("Recent Files"),
                                new FileIcon (icon_dir.get_child ("group-recent.svg")));
      categories.append (cat);

      cat =  new Unity.Category (_("Downloads"),
                                 new FileIcon (icon_dir.get_child ("group-downloads.svg")));
      categories.append (cat);

      cat = new Unity.Category (_("Folders"),
                                new FileIcon (icon_dir.get_child ("group-folders.svg")));
      categories.append (cat);

      cat = new Unity.Category (_("Files & Folders"),
                                new FileIcon (icon_dir.get_child ("group-folders.svg")));
      categories.append (cat);

      lens.categories = categories;
    }

    private void prepare_type_templates ()
    {
      type_templates = new HashTable<string, Event> (str_hash, str_equal);
      Event event;

      /* Section.ALL_FILES */
      event = new Event.full("", ZG_USER_ACTIVITY, "",
                             new Subject.full ("file:*",
                                               "", "", "", "", "", ""));
      type_templates["all"] = event;
      
      /* Section.DOCUMENTS */
      event = new Event.full("", ZG_USER_ACTIVITY, "",
                             new Subject.full ("file:*",
                                               NFO_DOCUMENT,
                                               "", "", "", "", ""),
                             new Subject.full ("file:*",
                                               "!"+NFO_PRESENTATION,
                                               "", "", "", "", ""));
      type_templates["documents"] = event;

      /* Section.FOLDERS
       * - we're using special ORIGIN queries here */
      event = new Event.full("", ZG_USER_ACTIVITY, "",
                             new Subject.full ("file:*",
                                               "", "", "", "", "", ""));
      type_templates["folders"] = event;

      /* Section.IMAGES */
      event = new Event.full("", ZG_USER_ACTIVITY, "",
                             new Subject.full ("file:*",
                                               NFO_IMAGE, "", "", "", "", ""));
      type_templates["images"] = event;
      
      /* Section.AUDIO */
      event = new Event.full("", ZG_USER_ACTIVITY, "",
                             new Subject.full ("file:*",
                                               NFO_AUDIO, "", "", "", "", ""));
      type_templates["audio"] = event;

      /* Section.VIDEOS */
      event = new Event.full("", ZG_USER_ACTIVITY, "",
                             new Subject.full ("file:*",
                                               NFO_VIDEO, "", "", "", "", ""));
      type_templates["videos"] = event;

      /* Section.PRESENTATIONS
       * FIXME: Zeitgeist logger needs to user finer granularity
       *        on classification as I am not sure it uses
       *        NFO_PRESENTATION yet */
      event = new Event.full("", ZG_USER_ACTIVITY, "",
                             new Subject.full ("file:*",
                                               NFO_PRESENTATION, "", "", "", "", ""));
      type_templates["presentations"] = event;

      /* Section.OTHER 
       * Note that subject templates are joined with logical AND */
      event = new Event.full("", ZG_USER_ACTIVITY, "");
      event.add_subject (new Subject.full ("file:*",
                                           "!"+NFO_DOCUMENT, "", "", "", "", ""));
      event.add_subject (new Subject.full ("",
                                           "!"+NFO_IMAGE,
                                           "",
                                           "", "", "", ""));
      event.add_subject (new Subject.full ("",
                                           "!"+NFO_AUDIO,
                                           "",
                                           "", "", "", ""));
      event.add_subject (new Subject.full ("",
                                           "!"+NFO_VIDEO,
                                           "",
                                           "", "", "", ""));
      event.add_subject (new Subject.full ("",
                                           "!"+NFO_PRESENTATION,
                                           "",
                                           "", "", "", ""));
      type_templates["other"] = event;
    }

    private const string[] ALL_TYPES =
    {
      "documents",
      "folders",
      "images",
      "audio",
      "videos",
      "presentations",
      "other"
    };

    private GenericArray<Event> create_template (OptionsFilter? filter)
    {
      var templates = new GenericArray<Event> ();

      if (filter == null || !filter.filtering)
      {
        /* Section.ALL_FILES */
        templates.add (type_templates["all"]);
        return templates;
      }

      string[] types = {};

      foreach (unowned string type_id in ALL_TYPES)
      {
        var option = filter.get_option (type_id);
        if (option == null || !option.active) continue;

        types += type_id;
      }

      if (types.length == ALL_TYPES.length)
      {
        /* Section.ALL_FILES */
        templates.add (type_templates["all"]);
        return templates;
      }

      foreach (unowned string type_id in types)
      {
        // we need to handle folders separately
        if (type_id == "folders") continue;

        templates.add (type_templates[type_id]);
      }

      return templates;
    }

    private bool is_search_empty (LensSearch search)
    {
      if (search.search_string == null) return true;
      
      return search.search_string.strip () == "";
    }

    private string prepare_search_string (LensSearch search)
    {
      var s = search.search_string;

      if (s.has_suffix (" "))
        s = s.strip ();

      if (!s.has_suffix ("*"))
        s = s + "*";

      /* The Xapian query parser (used by the Zeitgeist FTS Extension) seems to
       * handle hyphens in a special way, namely that it forces the joined
       * tokens into a phrase query no matter if it appears as the last word
       * in a query and we have the PARTIAL flag set on the query parser.
       * This makes 'unity-p' not match 'unity-package-search.cc' etc. */
      s = s.delimit ("-", ' ');

      return s;
    }
    
    private async void update_global_search_async (LensSearch search,
                                                   Cancellable cancellable)
    {
      var has_search = !is_search_empty (search);
      var results_model = search.results_model;

      /*
       * For global searches we collate all results under one category heading
       * called Files & Folders
       */

      try {
        /* Get results ranked by recency */
        var results = yield run_zg_query (search,
                                          new Zeitgeist.TimeRange.anytime (),
                                          ResultType.MOST_RECENT_SUBJECTS,
                                          null,
                                          20,
                                          cancellable);

        results_model.clear ();

        /* check if the thing typed isn't a url (like facebook.com) */
        var checked_url = urls.check_url (search.search_string);
        if (checked_url != null)
        {
          results_model.append (checked_url, urls.icon,
                                Categories.FILES_AND_FOLDERS,
                                "text/html", search.search_string,
                                checked_url, checked_url);
        }

        var category_id = has_search ?
          Categories.FILES_AND_FOLDERS : Categories.RECENT_FILES;

        Unity.FilesLens.append_events_sorted (results, results_model,
                                              0, int64.MAX, false,
                                              category_id);

        /* Add downloads catagory if we don't have a search */
        if (has_search == false)
        {
          yield update_downloads_async (results_model, cancellable,
                                        search.search_string);
        }

      } catch (IOError.CANCELLED ioe) {
        return;
      } catch (GLib.Error e) {
        warning ("Error performing global search '%s': %s",
                 search.search_string, e.message);
      }
    }
    
    private async void update_search_async (LensSearch search,
                                            Cancellable cancellable)
    {
      var results_model = search.results_model;
      var txn = new Dee.Transaction (results_model);
      var has_search = !is_search_empty (search);

      var filter = scope.get_filter ("type") as OptionsFilter;

      var active_filters = get_current_types ();
      bool only_folders = active_filters != null &&
        active_filters.length == 1 && active_filters[0] == "folders";
      uint timer_id = 0;

      try
      {
        /* Get results ranked by recency */
        ResultSet? results = null;
        if (!only_folders)
        {
          results = yield run_zg_query (search,
                                        get_current_timerange (),
                                        ResultType.MOST_RECENT_SUBJECTS,
                                        filter,
                                        50,
                                        cancellable);
        }

        txn.clear ();

        /* check if the thing typed isn't a url (like facebook.com) */
        var checked_url = urls.check_url (search.search_string);
        if (checked_url != null)
        {
          txn.append (checked_url, urls.icon, Categories.RECENT,
                      "text/html", search.search_string,
                      checked_url, checked_url);
        }

        /* apply filters to results found by zeitgeist */
        int64 min_size, max_size;
        get_current_size_limits (out min_size, out max_size);

        if (results != null)
        {
          Unity.FilesLens.append_events_sorted (results, txn,
                                                min_size, max_size,
                                                false);
        }

        /* get recently downloaded files */
        yield update_downloads_async (txn, cancellable,
                                      search.search_string);

        /* commit if the origin query is taking too long, if we committed right
         * away, we'd cause flicker */
        if (txn.get_n_rows () > 0)
        {
          /* here be something magical */
          timer_id = Timeout.add (200, () =>
          {
            if (!cancellable.is_cancelled () && !txn.is_committed ())
            {
              txn.commit ();
            }
            timer_id = 0;
            return false;
          });
        }

        /* folders are last category we need, update the model directly */
        if (filter == null ||
            !filter.filtering || filter.get_option ("folders").active)
        {
          results = yield run_zg_query (search, get_current_timerange (),
                                        ResultType.MOST_RECENT_ORIGIN,
                                        null, 50, cancellable);

          if (!txn.is_committed ()) txn.commit ();

          /* add bookmarks first */
          append_bookmarks (has_search ?
            bookmarks.prefix_search (search.search_string) : bookmarks.list (),
            results_model,
            Categories.FOLDERS);

          Unity.FilesLens.append_events_sorted (results, results_model,
                                                min_size, max_size,
                                                true);
        }
        else
        {
          /* just commit */
          if (!txn.is_committed ()) txn.commit ();
        }

      } catch (IOError.CANCELLED ioe) {
        return;
      } catch (GLib.Error e) {
        /* if we weren't cancelled, commit the transaction */
        warning ("Error performing global search '%s': %s",
                 search.search_string, e.message);
        if (!txn.is_committed ()) txn.commit ();
      } finally {
        if (timer_id != 0) Source.remove (timer_id);
      }
    }

    private string[]? get_current_types ()
    {
      /* returns null if the filter is disabled / all options selected */
      var filter = scope.get_filter ("type") as CheckOptionFilter;

      if (filter == null || !filter.filtering) return null;
      string[] types = {};

      foreach (unowned string type_id in ALL_TYPES)
      {
        var option = filter.get_option (type_id);
        if (option == null || !option.active) continue;

        types += type_id;
      }

      if (types.length == ALL_TYPES.length) return null;

      return types;
    }

    private TimeRange get_current_timerange ()
    {
      var filter = scope.get_filter ("modified") as RadioOptionFilter;
      Unity.FilterOption? option = filter.get_active_option ();

      string date = option == null ? "all" : option.id;

      if (date == "last-7-days")
        return new TimeRange (Timestamp.now() - Timestamp.WEEK, Timestamp.now ());
      else if (date == "last-30-days")
        return new TimeRange (Timestamp.now() - (Timestamp.WEEK * 4), Timestamp.now());
      else if (date == "last-year")
        return new TimeRange (Timestamp.now() - Timestamp.YEAR, Timestamp.now ());
      else
        return new TimeRange.anytime ();
    }

    private void get_current_size_limits (out int64 min_size, out int64 max_size)
    {
      var filter = scope.get_filter ("size") as MultiRangeFilter;
      Unity.FilterOption? min_opt = filter.get_first_active ();
      Unity.FilterOption? max_opt = filter.get_last_active ();

      if (min_opt == null || max_opt == null)
      {
        min_size = 0;
        max_size = int64.MAX;
        return;
      }

      int64[] sizes = 
      {
        0,
        1024,
        102400,
        1048576,
        10485760,
        104857600,
        1073741824,
        int64.MAX
      };

      switch (min_opt.id)
      {
        case "1KB":
          min_size = sizes[0]; break;
        case "100KB":
          min_size = sizes[1]; break;
        case "1MB":
          min_size = sizes[2]; break;
        case "10MB":
          min_size = sizes[3]; break;
        case "100MB":
          min_size = sizes[4]; break;
        case "1GB":
          min_size = sizes[5]; break;
        case ">1GB":
          min_size = sizes[6]; break;
        default:
          warn_if_reached ();
          min_size = 0;
          break;
      }

      switch (max_opt.id)
      {
        case "1KB":
          max_size = sizes[1]; break;
        case "100KB":
          max_size = sizes[2]; break;
        case "1MB":
          max_size = sizes[3]; break;
        case "10MB":
          max_size = sizes[4]; break;
        case "100MB":
          max_size = sizes[5]; break;
        case "1GB":
          max_size = sizes[6]; break;
        case ">1GB":
          max_size = sizes[7]; break;
        default:
          warn_if_reached ();
          max_size = int64.MAX;
          break;
      }
    }

    private async ResultSet run_zg_query (LensSearch search,
                                          TimeRange time_range,
                                          ResultType result_type,
                                          OptionsFilter? filters,
                                          uint num_results,
                                          Cancellable cancellable) throws Error
    {
      ResultSet results;
      var timer = new Timer ();
      var templates = create_template (filters);

      /* Copy the templates to a PtrArray which libzg expects */
      var ptr_arr = new PtrArray ();
      for (int i = 0; i < templates.length; i++)
      {
        ptr_arr.add (templates[i]);
      }

      /* Get results ranked by recency */
      if (is_search_empty (search))
      {
        results = yield log.find_events (time_range,
                                         (owned) ptr_arr,
                                         Zeitgeist.StorageState.ANY,
                                         num_results,
                                         result_type,
                                         cancellable);
      }
      else
      {
        var search_string = prepare_search_string (search);

        results = yield index.search (search_string,
                                      time_range,
                                      (owned) ptr_arr,
                                      0, // offset
                                      num_results,
                                      result_type,
                                      cancellable);
      }

      debug ("Found %u/%u no search results in %fms",
             results.size (), results.estimated_matches (),
             timer.elapsed()*1000);

      return results;
    }

    private void append_bookmarks (GLib.List<Bookmark> bookmarks,
                                   Dee.Model results_model,
                                   Categories category)
    {
      foreach (var bookmark in bookmarks)
      {
        results_model.append (bookmark.uri, bookmark.icon, category,
                              bookmark.mimetype, bookmark.display_name,
                              bookmark.dnd_uri);
      }
    }

    private async void update_downloads_async (Dee.Model results_model,
                                               Cancellable cancellable,
                                               string? name_filter = null,
                                               int category_override = -1) throws IOError
    {
      // FIXME: Store the Downloads folder and update on changes
      unowned string download_path =
                 Environment.get_user_special_dir (UserDirectory.DOWNLOAD);
      var download_dir = File.new_for_path (download_path);
      SList<FileInfo> downloads;

      try {
        if (name_filter != null && name_filter != "")
          downloads = yield Utils.list_dir_filtered (download_dir, name_filter);
        else
          downloads = yield Utils.list_dir (download_dir);
      } catch (GLib.Error e) {
        warning ("Failed to list downloads from directory '%s': %s",
                 download_path, e.message);
        return;
      }

      if (cancellable.is_cancelled ())
        throw new IOError.CANCELLED ("Search was cancelled");
      
      /* Sort files by mtime, we do an ugly nested ternary
       * to avoid potential long/int overflow */
      downloads.sort ((CompareFunc) Utils.cmp_file_info_by_mtime);
      
      var timerange = get_current_timerange ();
      int64 min_size, max_size;
      get_current_size_limits (out min_size, out max_size);
      
      foreach (var info in downloads)
      {
        var uri = download_dir.get_child (info.get_name ()).get_uri ();
        var mimetype = info.get_content_type ();
        var icon_hint = Utils.check_icon_string (uri, mimetype, info);

        // check if we match the timerange
        uint64 atime = info.get_attribute_uint64 (FILE_ATTRIBUTE_TIME_ACCESS) * 1000;
        
        if (atime < timerange.get_start() || atime > timerange.get_end ())
          continue;

        // check if type matches
        var types = get_current_types ();
        if (types != null && !Utils.file_info_matches_any (info, types))
          continue;
        
        // check if size is within bounds
        int64 size = info.get_size ();
        if (size < min_size || size > max_size)
          continue;
        
        uint category_id = Categories.DOWNLOADS;
        
        if (category_override >= 0)
          category_id = category_override;
        
        results_model.append (uri, icon_hint, category_id,
                              mimetype, info.get_display_name (), uri);
      }
    }

    public Unity.ActivationResponse activate (string uri)
    {
      debug (@"Activating: $uri");
      try {
        if (!bookmarks.launch_if_bookmark (uri))
          AppInfo.launch_default_for_uri (uri, null);
        return new Unity.ActivationResponse(Unity.HandledType.HIDE_DASH);
      } catch (GLib.Error error) {
        warning ("Failed to launch URI %s", uri);
        return new Unity.ActivationResponse(Unity.HandledType.NOT_HANDLED);
      }
    }

    private void on_zeitgeist_changed ()
    {
      /* make sure our results are fresh */
      scope.queue_search_changed (SearchType.DEFAULT);
      scope.queue_search_changed (SearchType.GLOBAL);
    }
  }

  private const string ATTR_HIDDEN = FILE_ATTRIBUTE_STANDARD_IS_HIDDEN;
  private const string ATTR_SIZE_AND_HIDDEN = FILE_ATTRIBUTE_STANDARD_SIZE +
    "," + FILE_ATTRIBUTE_STANDARD_IS_HIDDEN;

  /* Appends a set of Zeitgeist.Events to our Dee.Model assuming that
   * these events are already sorted with descending timestamps */
  public void append_events_sorted (Zeitgeist.ResultSet events,
                                    Dee.Model results,
                                    int64 min_size, int64 max_size,
                                    bool use_origin,
                                    int category_override = -1)
  {
    foreach (var ev in events)
    {
      if (ev.num_subjects() > 0)
      {
        // FIXME: We only use the first subject...
        Zeitgeist.Subject su = ev.get_subject(0);

        string uri;
        string display_name;
        string mimetype;

        if (use_origin)
        {
          uri = su.get_origin ();
          display_name = "";
          mimetype = "inode/directory";
        }
        else
        {
          uri = su.get_uri ();
          display_name = su.get_text ();
          mimetype = su.get_mimetype ();
          mimetype = su.get_mimetype () != null ?
                     su.get_mimetype () : "application/octet-stream";
        }
        if (uri == null) continue;
        File file = File.new_for_uri (uri);

        if (display_name == null || display_name == "")
        {
          display_name = Path.get_basename (file.get_parse_name ());
        }

        bool check_size = min_size > 0 || max_size < int64.MAX;
        /* Don't check existence on non-native files as http:// and
         * friends are *very* expensive to query */
        if (file.is_native()) {
          // hidden files should be ignored
          try {
            FileInfo info = file.query_info (check_size ?
              ATTR_SIZE_AND_HIDDEN : ATTR_HIDDEN, 0, null);
            if (info.get_is_hidden())
              continue;
            if (check_size &&
              (info.get_size () < min_size || info.get_size () > max_size))
            {
              continue;
            }
          } catch (GLib.Error e) {
            // as error occurred file must be missing therefore ignoring it
            continue;
          }
        }
        string icon = Utils.get_icon_for_uri (uri, mimetype);

        uint category_id;
        string comment = file.get_parse_name ();
        
        if (category_override >= 0)
          category_id = category_override;
        else
          category_id = file.query_file_type (0, null) == FileType.DIRECTORY ?
                                     Categories.FOLDERS : Categories.RECENT;
            
        results.append (uri, icon, category_id, mimetype,
                        display_name, comment);

      }
    }
  }
} /* namespace */

