/**
 * geis.c xcb-based GEIS implementation
 *
 * Copyright 2010 Canonical Ltd.
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */
#include <geis/geis.h>

#include "geis_instance.h"
#include "geis_xcb.h"
#include <stdlib.h>


static GeisXcb s_geis_xcb = NULL;


/**
 * @param[in]  win_info
 * @param[out] geis_instance
 *
 * @retval GEIS_BAD_ARGUMENT
 * @retval GEIS_STATUS_SUCCESS
 */
GeisStatus
geis_init(GeisWinInfo *win_info, GeisInstance *geis_instance)
{
  *geis_instance = NULL;
  GeisXcbWinInfo *xcb_win_info = (GeisXcbWinInfo*)win_info->win_info;

  if (win_info->win_type != GEIS_XCB_FULL_WINDOW)
  {
    return GEIS_BAD_ARGUMENT;
  }

  if (!s_geis_xcb)
  {
    s_geis_xcb = geis_xcb_new(xcb_win_info);
  }

  *geis_instance = geis_xcb_create_instance(s_geis_xcb, xcb_win_info->window_id);
  return GEIS_STATUS_SUCCESS;
}


/**
 * @param[in]  geis_instance
 */
GeisStatus
geis_finish(GeisInstance geis_instance)
{
  if (!geis_instance_is_valid(geis_instance))
  {
    return GEIS_BAD_ARGUMENT;
  }
  geis_instance_free(geis_instance);
  return GEIS_UNKNOWN_ERROR;
}


/**
 * @param[in]  geis_instance
 * @param[in]  configuration_item
 */
GeisStatus
geis_configuration_supported(GeisInstance geis_instance,
                             int          configuration_item)
{
  GeisStatus status = GEIS_STATUS_NOT_SUPPORTED;

  if (!geis_instance_is_valid(geis_instance))
  {
    return GEIS_BAD_ARGUMENT;
  }

  switch (configuration_item)
  {
    case GEIS_CONFIG_UNIX_FD:
      status = GEIS_STATUS_SUCCESS;
      break;
  }
  return status;
}


/**
 * @param[in]  geis_instance
 * @param[in]  configuration_item
 * @param[out] value
 */
GeisStatus
geis_configuration_get_value(GeisInstance  geis_instance,
                             int           configuration_item, 
                             void         *value)
{
  GeisStatus status = GEIS_STATUS_NOT_SUPPORTED;

  if (!geis_instance_is_valid(geis_instance))
  {
    return GEIS_BAD_ARGUMENT;
  }
  if (!value)
  {
    return GEIS_BAD_ARGUMENT;
  }

  switch (configuration_item)
  {
    case GEIS_CONFIG_UNIX_FD:
      *(int *)value = geis_xcb_fd(s_geis_xcb);
      status = GEIS_STATUS_SUCCESS;
      break;
  }
  return status;
}


/**
 * @param[in]  geis_instance
 * @param[in]  configuration_item
 * @param[in]  value
 */
GeisStatus
geis_configuration_set_value(GeisInstance  geis_instance,
                             int           configuration_item,
                             void         *value)
{
  if (!geis_instance_is_valid(geis_instance))
  {
    return GEIS_BAD_ARGUMENT;
  }

  return GEIS_UNKNOWN_ERROR;
}


/**
 * @param[in]  geis_instance
 */
GeisStatus
geis_event_dispatch(GeisInstance geis_instance)
{
  if (!geis_instance_is_valid(geis_instance))
  {
    return GEIS_BAD_ARGUMENT;
  }

  geis_xcb_dispatch(s_geis_xcb);
  return GEIS_STATUS_SUCCESS;
}


/**
 * @param[in]  geis_instance
 * @param[in]  funcs
 * @param[in]  cookie
 */
GeisStatus
geis_input_devices(GeisInstance    geis_instance,
                   GeisInputFuncs *funcs,
                   void           *cookie)
{
  if (!geis_instance_is_valid(geis_instance))
  {
    return GEIS_BAD_ARGUMENT;
  }

  return geis_xcb_input_devices(s_geis_xcb, funcs, cookie);
}


/**
 * @param[in]  geis_instance
 * @param[in]  input_list
 * @param[in]  gesture_list
 * @param[in]  callback
 * @param[in]  cookie
 */
GeisStatus
geis_subscribe(GeisInstance         geis_instance,
               GeisInputDeviceId   *input_list,
               const char*         *gesture_list,
               GeisGestureFuncs    *funcs,
               void                *cookie)
{
  if (!geis_instance_is_valid(geis_instance))
  {
    return GEIS_BAD_ARGUMENT;
  }

  return geis_instance_subscribe(geis_instance,
  				 s_geis_xcb,
                                 input_list,
                                 gesture_list,
                                 funcs,
                                 cookie);
}


/**
 * @param[in]  geis_instance
 * @param[in]  gesture_list
 */
GeisStatus
geis_unsubscribe(GeisInstance     geis_instance,
                 GeisGestureType *gesture_list)

{
  if (!geis_instance_is_valid(geis_instance))
  {
    return GEIS_BAD_ARGUMENT;
  }

  return GEIS_STATUS_SUCCESS;
}

