#include <linux/module.h>
#include <linux/kprobes.h>
#include <linux/ktime.h>
#include <linux/limits.h>
#include <linux/sched.h>
#include <linux/pci.h>
#include <linux/dmi.h>

#include <linux/version.h>
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3, 6, 0)
// for kernel 3.13
static char func_name[NAME_MAX] = "usb_enable_intel_xhci_ports";
#else
static char func_name[NAME_MAX] = "usb_enable_xhci_ports";
#endif


module_param_string(func, func_name, NAME_MAX, S_IRUGO);
MODULE_PARM_DESC(func, "route certain ports to EHCI instead of xHCI");

static u32 xusb2pr_value = -1;
module_param(xusb2pr_value , int, S_IRUGO|S_IWUSR);
MODULE_PARM_DESC(xusb2pr_value, "map ports to EHCI instead of xHCI");

// example:
// some usb bluetooth doesn't suppor USB3.0
// 
//  modprode xhci-quirk xusb2pr_value=0x01FF
//

static int matched = 0;
static u32 ports_available;

static int entry_handler(struct kretprobe_instance *ri, struct pt_regs *regs)
{
	return 0;
}

#define USB_INTEL_XUSB2PR	0xD0

static void set_xusb2pr(int restore)
{
	struct pci_dev *xhci_pdev;

	xhci_pdev = pci_get_subsys(PCI_VENDOR_ID_INTEL, 0x8c31, PCI_ANY_ID, PCI_ANY_ID, NULL);
    if (!xhci_pdev)
        xhci_pdev = pci_get_subsys(PCI_VENDOR_ID_INTEL, 0x9c31, PCI_ANY_ID, PCI_ANY_ID, NULL);

	if (!xhci_pdev) {
		printk(KERN_INFO "no device found\n");
		return;
	}
	if (restore) {
		pci_write_config_dword(xhci_pdev, USB_INTEL_XUSB2PR, ports_available);
		printk(KERN_INFO "restore set_xusb2pr orig = 0x%x\n", ports_available);
	} else {
		pci_read_config_dword(xhci_pdev, USB_INTEL_XUSB2PR, &ports_available);
		if (xusb2pr_value == -1) {
			xusb2pr_value=ports_available;
			printk(KERN_WARNING "[xhci-quirk] Need DMI information to modify xusb2pr_value");
		} else {
			pci_write_config_dword(xhci_pdev, USB_INTEL_XUSB2PR, xusb2pr_value);
			printk(KERN_INFO "old = 0x%x, new = 0x%x\n", ports_available, xusb2pr_value);
		}
	}
}

static int ret_handler(struct kretprobe_instance *ri, struct pt_regs *regs)
{
	set_xusb2pr(0);
	return 0;
}

static struct kretprobe my_kretprobe = {
	.handler		= ret_handler,
	.entry_handler		= entry_handler,
	/* Probe up to 20 instances concurrently. */
	.maxactive		= 20,
};

struct quirk_entry {
	u32 xusb2pr;
};

static struct quirk_entry *quirks;

/* caspian */
static struct quirk_entry quirk_dell_inspiron_3048 = {
	.xusb2pr = 0x0fdf,
};

/* cottonwood-intel-hsw */
static struct quirk_entry quirk_dell_inspiron_7347 = {
	.xusb2pr = 0x01df,
};

/* general-sff */
static struct quirk_entry quirk_dell_inspiron_3847 = {
	.xusb2pr = 0x0fdf,
};

/* pandora-intel-hsw */
static struct quirk_entry quirk_dell_inspiron_7447 = {
	.xusb2pr = 0x3cdF,
};

/* houston-12-intel-hsw */
static struct quirk_entry quirk_dell_latitude_5250 = {
	.xusb2pr = 0x01db,
};

/* marble-falls-15 */
static struct quirk_entry quirk_dell_latitude_3450 = {
	.xusb2pr = 0x06f,
};

/* thun */
static struct quirk_entry quirk_dell_inspiron_5348 = {
	.xusb2pr = 0x37ff,
};

/* thun-lite */
static struct quirk_entry quirk_dell_vostro_23_3340 = {
	.xusb2pr = 0x37ff,
};


static int dmi_matched(const struct dmi_system_id *dmi)
{

	matched = 1;

	quirks = dmi->driver_data;
    xusb2pr_value = quirks->xusb2pr;
	return 1;
}

static const struct dmi_system_id xhci_quirk_table[] = {
	{
		.callback = dmi_matched,
		.matches = {
			DMI_MATCH(DMI_PRODUCT_NAME, "Inspiron 20 Model 3048"),
		},
 		.driver_data = &quirk_dell_inspiron_3048,
	},
	{
		.callback = dmi_matched,
		.matches = {
			DMI_MATCH(DMI_PRODUCT_NAME, "Inspiron 23 Model 5348"),
		},
 		.driver_data = &quirk_dell_inspiron_5348,
	},
	{
		.callback = dmi_matched,
		.matches = {
			DMI_MATCH(DMI_PRODUCT_NAME, "Vostro 23-3340"),
		},
 		.driver_data = &quirk_dell_vostro_23_3340,
	},
	{
		.callback = dmi_matched,
		.matches = {
			DMI_MATCH(DMI_PRODUCT_NAME, "Inspiron 7347"),
		},
 		.driver_data = &quirk_dell_inspiron_7347,
	},
	{
		.callback = dmi_matched,
		.matches = {
			DMI_MATCH(DMI_PRODUCT_NAME, "Inspiron 3847"),
		},
 		.driver_data = &quirk_dell_inspiron_3847,
	},
	{
		.callback = dmi_matched,
		.matches = {
			DMI_MATCH(DMI_PRODUCT_NAME, "Inspiron 7447"),
		},
 		.driver_data = &quirk_dell_inspiron_7447,
	},
	{
		.callback = dmi_matched,
		.matches = {
			DMI_MATCH(DMI_PRODUCT_NAME, "Latitude 5250"),
		},
 		.driver_data = &quirk_dell_latitude_5250,
	},
	{
		.callback = dmi_matched,
		.matches = {
			DMI_MATCH(DMI_PRODUCT_NAME, "Latitude E5250"),
		},
 		.driver_data = &quirk_dell_latitude_5250,
	},
	{
		.callback = dmi_matched,
		.matches = {
			DMI_MATCH(DMI_PRODUCT_NAME, "Latitude 5450"),
		},
 		.driver_data = &quirk_dell_latitude_5250,
	},
	{
		.callback = dmi_matched,
		.matches = {
			DMI_MATCH(DMI_PRODUCT_NAME, "Latitude E4250"),
		},
 		.driver_data = &quirk_dell_latitude_5250,
	},
	{
		.callback = dmi_matched,
		.matches = {
			DMI_MATCH(DMI_PRODUCT_NAME, "Latitude 5550"),
		},
 		.driver_data = &quirk_dell_latitude_5250,
	},
	{
		.callback = dmi_matched,
		.matches = {
			DMI_MATCH(DMI_PRODUCT_NAME, "Latitude E5550"),
		},
 		.driver_data = &quirk_dell_latitude_5250,
	},	
	{
		.callback = dmi_matched,
		.matches = {
			DMI_MATCH(DMI_PRODUCT_NAME, "Latitude 3450"),
		},
 		.driver_data = &quirk_dell_latitude_3450,
	},
	{ }
};


static int __init kretprobe_init(void)
{
	int ret;
	my_kretprobe.kp.symbol_name = func_name;

	ret = register_kretprobe(&my_kretprobe);
	if (ret < 0) {
		printk(KERN_INFO "register_kretprobe failed, returned %d\n",
				ret);
		return -1;
	}

	dmi_check_system(xhci_quirk_table);

	set_xusb2pr(0);
	printk(KERN_INFO "Planted return probe at %s: %p\n",
			my_kretprobe.kp.symbol_name, my_kretprobe.kp.addr);
	return 0;
}

static void __exit kretprobe_exit(void)
{
	if (!matched)
		return;

	printk(KERN_INFO "Restore orignal XHCI mapping %x\n",
			ports_available);

	set_xusb2pr(1);

	unregister_kretprobe(&my_kretprobe);

	printk(KERN_INFO "kretprobe at %p unregistered\n",
			my_kretprobe.kp.addr);

	/* nmissed > 0 suggests that maxactive was set too low. */
	printk(KERN_INFO "Missed probing %d instances of %s\n",
		my_kretprobe.nmissed, my_kretprobe.kp.symbol_name);

}

module_init(kretprobe_init)
module_exit(kretprobe_exit)
MODULE_LICENSE("GPL");
/* Broadcom */
MODULE_ALIAS("pci:v000014E4d*sv*sd*bc*sc*i*");
/* Qualcomm Atheros */
MODULE_ALIAS("pci:v0000168Cd*sv*sd*bc*sc*i*");
